//
//  iWeb - iWHtmlControls.js
//  Copyright (c) 2006-2008 Apple Inc. All rights reserved.
//

//  Responsibility: brentm
//  Reviewers: 
//

// Controls that can be used in widgets and configurators.
//
// functions passed as event handlers are called in the context of the 
// JavaScript object, not the HTML object.
//

// Slider: WebKit-only slider control.
//
//
var Slider = Class.create({
    initialize: function(title, id, min, max, initialValue, onInput, showValue)
    {
        this.title = title;
        this.id = id;
        this.min = min;
        this.max = max;
        this.onInput = onInput;
        this.showValue = showValue;
    
        this.element = document.createElement("span");
        this.element.appendChild(document.createTextNode(this.title + (showValue ? ": " : "")));

        if (this.showValue)
        {
            this.valueSpan = document.createElement("span");
            this.element.appendChild(this.valueSpan);
            this.element.appendChild(document.createElement("br"));
        }
        this.input = document.createElement("input");
        this.element.appendChild(this.input);
        this.input.setAttribute("id", this.id);
        this.input.setAttribute("type", "range");
        this.input.setAttribute("min", 0);
        this.input.setAttribute("max", 100);
        this.setValue(initialValue);
    
        var self = this;
        this.input.oninput = function()
        {
            self.setValue(self.sliderValueFromControlValue(this.value));
        };
        if (document.slider === undefined)
        {
            document.slider = {};
        }
        document.slider[id] = this;
    },

    setValue: function(value)
    {
        value = Math.min(Math.max(this.min, value), this.max);

        if (this.value !== value)
        {
            var callHandler = (this.value !== undefined);
            this.value = value;
            this.input.value = this.controlValueFromSliderValue(value);
            if (this.showValue)
            {
                this.valueSpan.innerText = value;
            }
            callHandler && this.onInput();
        }
    },

    // Note: The slider thumb only travels between 0 and 100 on the control, so setting min/max values
    // outside [0,100] results in display bugs because the thumb gets 'stuck'. So, we always use a range
    // of [0,100] and scale to the desired range.
    // Also: I haven't found a way to change the displayed width of a slider.
    //
    sliderValueFromControlValue: function(controlValue)
    {
        return Math.round((controlValue / 100.0) * (this.max - this.min) + this.min);
    },

    controlValueFromSliderValue: function(sliderValue)
    {
        return Math.round(((sliderValue - this.min) / (this.max - this.min)) * 100);
    }
});


// Checkbox
//
// Uses 'checked' property of <input> element as control state.
//

function coerceToBoolean(b)
{
    if ((b === true) || (b === false))
        return b;
        
    if (b == "true") return true;
    if ((typeof b == "number") && (b !== 0)) return true;
    return false;
}

var Checkbox = Class.create({
    initialize: function(title, id, initialValue, onChange)
    {
        this.title = title;
        this.id = id;
        this.onChange = onChange;
    
        this.element = document.createElement("span");
        this.input = document.createElement("input");
        this.element.appendChild(this.input);
        this.element.appendChild(document.createTextNode(" " + this.title));
        this.input.id = this.id;
        this.input.setAttribute("type", "checkbox");

        this.setValue(initialValue);

        var self = this;
        this.input.onchange = function()
        {
            self.value = this.checked;
            self.onChange();
        };
        if (document.checkbox === undefined)
        {
            document.checkbox = {};
        }
        document.checkbox[id] = this;
    },

    setValue: function(newValue)
    {
        newValue = coerceToBoolean(newValue);
        if (this.value != newValue)
        {
            var callHandler = (this.value !== undefined);
            this.input.checked = newValue;
            callHandler && this.onChange();
        }
    }
});


var Popup = Class.create({
    initialize: function(title, id, values, onChange )
    {
        this.title = title;
        this.id = id;
        this.onChange = onChange;
    
        this.element = document.createElement("span");
        if (title.length >0)
        {
            this.element.appendChild(document.createTextNode(this.title + ": " ));
        }
        this.select = document.createElement("select");
        this.select.id = id;
    
        for (var i = 0; i < values.length; ++i)
        {
            if (i == 0)
            {
                this.value = values[i];
            }
            var item = createElementWithText("option", values[i]);
            this.select.appendChild(item);
        }
    
        this.element.appendChild(this.select);
    
        var self = this;
        this.select.onchange = function()
        {
            self.value = this.value;
            self.onChange();
        }
        if (document.popup === undefined)
        {
            document.popup = {};
        }
        document.popup[id] = this;
    },

    setValue: function(newValue)
    {
        if (this.value != newValue)
        {
            this.select.value = newValue;
        }
    }
});

var ImageButton = Class.create({
    initialize: function(id, url, width, onClick)
    {
        this.id = id;
        this.onClick = onClick;
        this.imageUrl = url;
    
        this.element = document.createElement("img");
        this.element.src = this.imageUrl;
        this.element.style.width = px(width);

        var self = this;
        this.element.onclick = function()
        {
            self.onClick();
        };
        if (document.imageButton === undefined)
            document.imageButton = {};
        
        document.imageButton[id] = this;
    }
});

var Button = Class.create({
    initialize: function(title, id, onClick)
    {
        this.id = id;
        this.onClick = onClick;
    
        this.element = document.createElement("span");
        this.input = document.createElement("input");
        this.element.appendChild(this.input);
        this.input.value = title;
        this.input.type = "button";

        var self = this;
        this.input.onclick = function()
        {
            self.onClick();
        }
        if (document.button === undefined)
            document.button = {};
        document.button[id] = this;
    }
});

var TextField = Class.create({
    initialize: function(title, id, width)
    {
        this.id = id;
        this.element = document.createElement("span");
        this.input = document.createElement("input");
        this.input.size = width;
        this.element.appendChild(this.input);
    
        if(document.textfield === undefined)
            document.textfield = {};
        document.textfield[id] = this;

    }
});

function createElementWithText(tagName, text)
{
    var element = document.createElement(tagName);
    var textNode = document.createTextNode(text);
    element.appendChild(textNode);
    return element;
}

