//
// iWeb - iWebBlog.js
// Copyright 2007-2008 Apple Inc.
// All rights reserved.
//


// @import iWebSite.js
// @import https://configuration.apple.com/configurations/internetservices/iapps/publishConfiguration08.plist?locationSearchJavascript

//<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<

// Returns a string that is the URL of the root folder of the blog that contains the
// provided page URL. Works for main, archive and entry pages in a blog.
//
function BlogRootURLString(inUrlString)
{
    var urlString = inUrlString.urlStringByDeletingQueryAndFragment();
    var index = urlString.search(/\/\d{4}\/\d{1,2}\//);
    if (index != -1)
    {
        return urlString.substr(0, index).stringByDeletingLastPathComponent();
    }
    return urlString.substr(0, urlString.lastIndexOf("/"));
}

function BlogRootURL(inUrlString)
{
    return new IWURL(BlogRootURLString(inUrlString));
}


// Call from onpageload to start process of loading archive feed and fixing up blog prev/next links
//
function BlogFixupPreviousNext()
{
    var currentUrl = locationHRef().urlStringByDeletingQueryAndFragment();
    window.blogFeed = new BlogFeed(BlogRootURLString(locationHRef()), true, function(blogFeed)
    {
        var prevNextLinks = $$(".iWebBlogPrev", ".iWebBlogNext");
        prevNextLinks.each(function(anchor)
        {
            var targetItem = null;
            if (anchor.hasClassName("iWebBlogPrev"))
            {
                targetItem = blogFeed.itemBefore(currentUrl);
            }
            else if (anchor.hasClassName("iWebBlogNext"))
            {
                targetItem = blogFeed.itemAfter(currentUrl);
            }
            if (targetItem)
            {
                anchor.href = targetItem.absoluteURL.toURLString();
                anchor.title = targetItem.title;
            }
        });
    });
}

function BlogPreviousPage()
{
    var currentUrlString = locationHRef().urlStringByDeletingQueryAndFragment();
    var targetUrlString = window.blogFeed.itemBefore(currentUrlString);
    location.href = targetUrlString;
}

function BlogNextPage()
{
    var currentUrlString = locationHRef().urlStringByDeletingQueryAndFragment();
    var targetUrlString = window.blogFeed.itemAfter(currentUrlString);
    location.href = targetUrlString;
}

function BlogMainPageItem()
{
    var blogURLString = BlogRootURLString(locationHRef());
    if (window.iWebBlogMainPageName === undefined)
    {
        window.iWebBlogMainPageTitle = blogURLString.lastPathComponent();
        window.iWebBlogMainPageName = window.iWebBlogMainPageTitle + ".html";
    }
    
    blogURLString = blogURLString.stringByAppendingPathComponent(window.iWebBlogMainPageName);
    return { absoluteURL: new IWURL(blogURLString), title: window.iWebBlogMainPageTitle };
}

function BlogArchivePageItem()
{
    var blogURLString = BlogRootURLString(locationHRef());
    if (window.iWebBlogArchivePageName === undefined)
    {
        window.iWebBlogArchivePageTitle = "Archive";
        window.iWebBlogArchivePageName = window.iWebBlogArchivePageTitle + ".html";
    }
    blogURLString = blogURLString.stringByAppendingPathComponent(window.iWebBlogArchivePageName);
    return { absoluteURL: new IWURL(blogURLString), title: window.iWebBlogArchivePageTitle };
}



//
// The BlogFeed object has the following data structure:
//
// this.mItems[] = array of item
// this.mBaseURL = absolute URL of blog (IWURL)
// this.mDateFormat = ICU date format string
// this.mMaximumSummaryItems = integer
// this.mIsArchive = true or false
//
// item = {
//   title = string
//   absoluteURL = URL of item
//   date  = date
//   dateString = string   // will use string if we have it
//   description = richText
//   imageUrlString = url string
//   p_linkText  = url string read from feed(deprecated)
// }

var BlogFeed = Class.create({
    initialize: function(inBlogUrlString, inIsArchive, inCallback)
    {
        this.mBlogURL = new IWURL(inBlogUrlString);
        this.mIsArchive = inIsArchive;
    
        var feedUrlString = inIsArchive ? "blog-archive.xml" : "blog-main.xml";
            feedUrlString = inBlogUrlString.stringByAppendingPathComponent(feedUrlString);

        delete this.mItems;

        new Ajax.Request(feedUrlString, {
            method: 'get',
            onSuccess: function(request) {
                this.p_parseFeed(ajaxGetDocumentElement(request));
            }.bind(this),
            onComplete: function()
            {
                inCallback(this);
            }.bind(this)
        });
    },

    p_parseFeed: function(rssDoc)
    {
        this.mDateFormat = "EEEE, MMMM d, yyyy";
        this.mBaseURL = new IWURL();
        this.mMaximumSummaryItems = 10;
        this.mItems = [];

        var channel = rssDoc.getElementsByTagName("channel")[0];

        var dateFormat          = BlogFeed.getiWebElementText(channel, "dateFormat");
        var maximumSummaryItems = BlogFeed.getiWebElementText(channel, "maximumSummaryItems");
        var baseURLString       = BlogFeed.getiWebElementText(channel, "baseURL");

        if (dateFormat)
            this.mDateFormat = dateFormat;
        if (maximumSummaryItems)
            this.mMaximumSummaryItems = maximumSummaryItems;

        if (baseURLString)
        {
            this.mBaseURL = new IWURL(baseURLString);
        }

        var itemNodes = channel.getElementsByTagName("item");
        for (var i = 0; i < itemNodes.length; ++i)
        {
            var itemNode = itemNodes[i];
            var item;
            try
            {
                item = new BlogFeed.FeedItem(itemNode);
                if (this.mBaseURL)
                {
                    // Compute relative URL of this item.
                    // Rebase absoluteURL to the actual URL of the entry in this blog.
                    item.relativeURL = item.absoluteURL.relativize(this.mBaseURL);
                    item.absoluteURL = item.relativeURL.resolve(this.mBlogURL);

                    if (item.commentURL)
                    {
                        item.relativeCommentURL = item.commentURL.relativize(this.mBaseURL);
                    }
                }
                this.mItems.push(item);
            }
            catch (e)
            {
                debugPrintException(e);
            }
        }
    },

    // Return the number of items in the feed.
    //
    itemCount: function()
    {
        if (this.mItems === undefined)
            return 0;
        return this.mItems.length;
    },

    // Return the FeedItem at the specified index
    //
    itemAtIndex: function(index)
    {
        return this.mItems[index];
    },

    // Return the date format to be used for the feed
    //
    dateFormat: function()
    {
        return this.mDateFormat;
    },

    // Return the maximum number of items to display for the feed.
    //
    maximumItemsToDisplay: function()
    {
        var result = this.itemCount();
        if (!this.mIsArchive && (this.mMaximumSummaryItems > 0) && (this.mMaximumSummaryItems < result))
        {
            result = this.mMaximumSummaryItems;
        }
        return result;
    },

    dumpFeed: function()
    {
        print("dumping a feed with %s items", this.itemCount());
        for (var i = 0; i < this.itemCount(); ++i)
        {
            printObject(this.itemAtIndex(i));
        }
    },

    itemAfter: function(urlString)
    {
        var afterIndex = null;
        var url = new IWURL(urlString);

        for (var i = 0; i < this.mItems.length; ++i)
        {
            if (url.isEqual(this.mItems[i].absoluteURL))
            {
                afterIndex = i - 1;
                break;
            }
        }
        if (afterIndex < 0)
            return BlogArchivePageItem();
        else
            return this.mItems[afterIndex];
    },

    itemBefore: function(urlString)
    {
        var beforeIndex = null;
        var url = new IWURL(urlString);

        for (var i = 0; i < this.mItems.length; ++i)
        {
            if (url.isEqual(this.mItems[i].absoluteURL))
            {
                beforeIndex = i + 1;
                break;
            }
        }
        if (beforeIndex < this.mItems.length)
            return this.mItems[beforeIndex];
        else
            return BlogMainPageItem();
    }
});


Object.extend(BlogFeed, {
    
    iwebNS: "http://www.apple.com/iweb",

    getiWebElement: function(itemNode, propertyName)
    {
        return getFirstChildElementByTagNameNS(itemNode, BlogFeed.iwebNS, "iweb", propertyName);
    },

    getiWebElementText: function(itemNode, propertyName)
    {
        return getChildElementTextByTagNameNS(itemNode, BlogFeed.iwebNS, "iweb", propertyName);
    },
    
    fixupURL: function(url)
    {
        // Fixup file URLs - NSURL produces file URLs with 'localhost' in the path, but
        // browsers don't include it.
        return url.replace("file://localhost/", "file:///");
    },
    
    FeedItem: function(itemNode)
    {
        var child = itemNode.firstChild;
        
        while (child)
        {
            if (child.nodeType == Node.ELEMENT_NODE)
            {
                if (child.tagName == 'title')
                {
                    this.title = getTextFromNode(child);
                }
                else if (child.tagName == 'link')
                {
                    this.p_linkText = getTextFromNode(child);
                    this.absoluteURL = new IWURL(this.p_linkText);
                }
                else if (child.tagName == 'description')
                {
                    this.description = getTextFromNode(child);
                }
                else if (child.tagName == 'pubDate')
                {
                    var dateText = getTextFromNode(child);
                    if (dateText && dateText.length > 0)
                        this.date = new Date(dateText);
                }
                else if (child.tagName == 'iweb:image')
                {
                    this.imageUrlString = child.getAttribute("href");
                    this.imageURL = new IWURL(this.imageUrlString);
                }
                else if (child.tagName == 'iweb:comment') // in-app only
                {
                    this.commentCount      = child.getAttribute("count");
                    this.commentingEnabled = (child.getAttribute("enabled") == 1);
                    this.commentURL        = new IWURL(child.getAttribute("link"));
                }
                else if (child.tagName == 'iweb:badge')
                {
                    iWLog("child.tagName='iweb:badge' type='" + child.getAttribute("type") + "'");
                    this.badgeType = child.getAttribute("type");
                }
            }
            child = child.nextSibling;
        }
        
        this.title = this.title || "";
        this.absoluteURL = this.absoluteURL || new IWURL();
        this.date = this.date || new Date();
        this.commentingEnabled = this.commentingEnabled || false;
        this.commentCount = this.commentCount || 0;
    }
});


//>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

//<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<

function iWebInitSearch()
{
    try
    {
        setLocale();
        initSearch();
    }
    catch (e)
    {
    }
}

//>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>


//<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<

// iWeb Dynamic Content Management
//

// Load and parse the content XML file for the current page,
// and call the appropriate methods to dynamically populate
// the HTML from the content.
function dynamicallyPopulate()
{
    // Load the content XML file for the current page.
    var contentXml = getContentXmlURL();
    if(contentXml == null)
    {
        // If not passed the content xml location via a query string, then contruct it ourselves.
        // Assume that the file has the same name, but with "xml" rather than "html" extension.
        var baseUrl = String(this.location).urlStringByDeletingQueryAndFragment();
        contentXml = String(baseUrl).replace(/\.html$/, '.xml');
    }

    // Callback to handle the AJAX request.
    var populateDomWithContentFromXML = function(request)
    {
        var contentDoc = ajaxGetDocumentElement(request);

        // Process each textBox element.
        $A(contentDoc.getElementsByTagName('textBox')).each(function(textbox) {
            try {
                dynamicallyPopulateTextBox(textbox);
            } catch (e) {
            }  
        });
        
        // Process each image element.
        $A(contentDoc.getElementsByTagName('image')).each(function(image) {
            try {
                dynamicallyPopulateImage(image);
            } catch (e) {
            }  
        });
    };

    // Load the content XML for the page.
    new Ajax.Request(contentXml, {
        method: 'get',
        onSuccess: populateDomWithContentFromXML
    });
}

// Given a textBox element from the content XML, populate the corresponding HTML element.
function dynamicallyPopulateTextBox(textBoxElement)
{
    if (textBoxElement)
    {
        var id = textBoxElement.getAttribute('id');
        var htmlElement = $(id);
        if (htmlElement)
        {
            var htmlParent = htmlElement.parentNode;
            if (textBoxElement.getAttribute('visible') == 'yes')
            {
                if (textBoxElement.getAttribute('dynamic') == 'yes')
                {
                    // Obtain the content from the richText element.  Should be escaped XHTML.
                    var content = String(getChildElementTextByTagName(textBoxElement, 'richText'));
            
                    // Inject the content into the html element.
                    htmlElement.innerHTML = content;

                    // Delete sibling elements (original content).
                    htmlParent.innerHTML = htmlElement.outerHTML;
                }

                // Regardless of whether or not we dynamically populated the content,
                // we need to display the parent, which was published hidden.
                htmlParent.style.visibility = 'visible';
            }
        }
    }
}

function dynamicallyPopulateImage(imageElement)
{
    if (imageElement)
    {
        var id = imageElement.getAttribute('id');
        var htmlElement = $(id);
        if (htmlElement)
        {
            if (imageElement.getAttribute('visible') == 'yes')
            {
                if (imageElement.getAttribute('dynamic') == 'yes')
                {
                    // Modify the HTML image properties based on the values from
                    // the content XML.
                    htmlElement.src = imageElement.getAttribute('src');
                    htmlElement.setStyle({
                        left: imageElement.getAttribute('left'),
                        top: imageElement.getAttribute('top'),
                        width: imageElement.getAttribute('width'),
                        height: imageElement.getAttribute('height')
                    });
                }

                // Regardless of whether or not we dynamically populated the content,
                // we need to display the HTML element, which was published hidden.
                htmlElement.style.visibility = 'visible';
            }
        }
    }
}

function getContentXmlURL()
{
    var params = locationHRef().toQueryParams();
    return params['content'];
}

//>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

//<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<

var HTMLTextModel = Class.create({
    initialize: function(node)
    {
        // Create a pair of arrays of character indices and the nodes that contain them. See nodeIndexForCharacterIndex,
        // but basically the text node at nodeArray[i] contains characters in the range indexArray[i-1]..indexArray[i]-1
        //
        this.indexArray = []; // index of last + 1 character...
        this.nodeArray = [];  // .. in this text node
    
        this.cachedNode = null;
        this.cachedNodeStart = null;
        this.cachedNodeEnd   = null;
    
        this.rootNode = node;
    
        this.buildTextModel(node);
    },

    buildTextModel: function(node)
    {
        for (var i = 0; i < node.childNodes.length; ++i)
        {
            var childNode = node.childNodes[i];
            if (childNode.nodeType == Node.TEXT_NODE)
            {
                this.nodeArray.push(childNode);
                this.indexArray.push((this.indexArray.length === 0 ? 0 : this.indexArray[this.indexArray.length - 1]) + childNode.nodeValue.length);
                this.length = this.indexArray[this.indexArray.length - 1];
            }
            else if (childNode.nodeType == Node.ELEMENT_NODE)
            {
                this.buildTextModel(childNode);
            }
        }
    },

    nodeIndexForCharacterIndex: function(index)
    {
        var nodeIndex;
    
        for (var i = 0; i < this.nodeArray.length; ++i)
        {
            if (index < this.indexArray[i])
            {
                nodeIndex = i;
                break;
            }
        }
        return nodeIndex;
    },

    characterAtIndex: function(index)
    {
        if ((this.cachedNode == null) || (index < this.cachedNodeStart) || (index >= this.cachedNodeEnd))
        {
            this.cachedNode = null;
            var nodeIndex = this.nodeIndexForCharacterIndex(index);
            if (nodeIndex !== undefined)
            {
                this.cachedNode = this.nodeArray[nodeIndex];
                this.cachedNodeStart = nodeIndex === 0 ? 0 : this.indexArray[nodeIndex - 1];
                this.cachedNodeEnd = this.cachedNodeStart + this.cachedNode.nodeValue.length;
            }
        }
        if (this.cachedNode)
        {
            return this.cachedNode.nodeValue.charAt(index - this.cachedNodeStart);
        }
        return "";
    },

    truncateAtIndex: function(index, suffix)
    {
        var nodeIndex = this.nodeIndexForCharacterIndex(index);
        if (nodeIndex !== undefined)
        {
            // Find the node with the index, truncate the text and add the suffix
            var node = this.nodeArray[nodeIndex];
            var subIndex = index - (nodeIndex === 0 ? 0 :  this.indexArray[nodeIndex - 1]);
            node.nodeValue = node.nodeValue.substr(0, subIndex);
            if (suffix)
            {
                node.nodeValue = node.nodeValue + suffix;
            }
       
            // Now remove all nextSiblings of node and its parents up to the root node.
            while (node != this.rootNode)
            {
                while (node.nextSibling != null)
                {
                    node.parentNode.removeChild(node.nextSibling);
                }
                node = node.parentNode;
            }
        }
    },

    truncateAroundPosition: function(index, suffix)
    {
        var isWordBreakCharacter = function(oneCharString)
        {
            // 0x0020 space
            // 0x000a line feed
            // 0x000d carriage return
            // 0x00a0 no-break space
            // 0x1680 ogham space mark
            // 0x2000-0x200b various unicode space characters
            // 0x2014 em dash
            // 0x202f narrow no-break space
            // 0x205f medium math space
            // 0x3000 ideographic space
            //
            if (oneCharString == null)
            {
                return false;
            }
            var ch = oneCharString.charCodeAt(0);
            return (ch == 0x0020) || (ch == 0x000a) || (ch == 0x000d) || (ch == 0x00a0) ||
                   (ch == 0x1680) || (ch >= 0x2000  &&  ch <= 0x200b) || (ch == 0x202f) ||
                   (ch == 0x205f) || (ch == 0x3000) || (ch == 0x2014);
        };
        
        var pos = index;
        if (pos >= this.length)
        {
            return;
        }
    
        while (pos >= 0)
        {
            if (isWordBreakCharacter(this.characterAtIndex(pos)))
            {
                while (pos > 0 && isWordBreakCharacter(this.characterAtIndex(pos)))
                {
                    pos--;
                }
                pos++;
                this.truncateAtIndex(pos, suffix);
                return;
            }
            pos--;
        }

        // If we got all the way back to zero without a word break then truncate violently.
        this.truncateAtIndex(index, suffix);
    }
});

//>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

//<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<

Date.prototype.ampmHour = function()
{
    // hour based on a 12-hour clock as a decimal number (01-12)
    var hour = this.getHours();
    if (hour >= 12)  hour -= 12;
    if (hour == 0) hour = 12;
    return hour;
}

// String Tables for mapping indices to strings.

Date.abbreviatedMonthNameKeys = 
["Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"];

Date.abbreviatedWeekdayNameKeys = 
["Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"];

Date.ampmKeys = ["AM", "PM" ];

Date.fullMonthNameKeys = 
["January", "February", "March", "April", "May", "June", "July", "August", "September", "October", "November", "December"];

Date.fullWeekdayNameKeys = 
["Sunday", "Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday"];


// If a localizer is provided, return the localized version of the string from the specified
// index of the table. Otherwise, just return the string.
//
// A localizer is any object that implements localizedString(s), but is usually a widget.
//
Date.localizedTableLookup = function(table, index, localizer)
{
    var string = table[index];
    if (localizer)
    {
        string = localizer.localizedString(string);
    }
    return string;
}

Date.abbreviatedMonthName = function(index, localizer)
{
    return Date.localizedTableLookup(Date.abbreviatedMonthNameKeys, index, localizer);
}

Date.abbreviatedWeekdayName = function(index, localizer)
{
    return Date.localizedTableLookup(Date.abbreviatedWeekdayNameKeys, index, localizer);
}

Date.ampmName = function(index, localizer)
{
    return Date.localizedTableLookup(Date.ampmKeys, index, localizer);
}

Date.fullWeekdayName = function(index, localizer)
{
    return Date.localizedTableLookup(Date.fullWeekdayNameKeys, index, localizer);
}

Date.fullMonthName = function(index, localizer)
{
    return Date.localizedTableLookup(Date.fullMonthNameKeys, index, localizer);
}

// This isn't a proper ICU date formatter, but will give the right results with formats currently
// supported in iWeb. A good exercise to generalize this and combine with the C style date
// formatting above. See http://icu.sourceforge.net/userguide/formatDateTime.html.
//
var formatConversion = {
//    "G1":"", not used
    "y1":function(d, localizer) { return d.getFullYear(); },
    "y2":function(d, localizer) { return ("0" + d.getFullYear()).slice(-2); },
    "y4":function(d, localizer) { return ("0000" + d.getFullYear()).slice(-4); },
    "M4":function(d, localizer) { return Date.fullMonthName(d.getMonth(), localizer); },
    "M3":function(d, localizer) { return Date.abbreviatedMonthName(d.getMonth(), localizer); },
    "M2":function(d, localizer) { return ("0" + (d.getMonth() + 1)).slice(-2); },
    "M1":function(d, localizer) { return String(d.getMonth() + 1); },
    "d2":function(d, localizer) { return ("0" + d.getDate()).slice(-2); },
    "d1":function(d, localizer) { return d.getDate(); },
    "h2":function(d, localizer) { return ("0" + d.ampmHour()).slice(-2); },
    "h1":function(d, localizer) { return d.ampmHour(); },
    "H2":function(d, localizer) { return ("0" + d.getHours()).slice(-2); },
    "H1":function(d, localizer) { return d.getHours(); },
    "m2":function(d, localizer) { return ("0" + d.getMinutes()).slice(-2); },
    "m1":function(d, localizer) { return d.getMinutes(); },
    "s2":function(d, localizer) { return ("0" + d.getSeconds()).slice(-2); },
    "s1":function(d, localizer) { return d.getSeconds(); },
// "S":"" not used    
    "E4":function(d, localizer) { return Date.fullWeekdayName(d.getDay(), localizer); },
    "E3":function(d, localizer) { return Date.abbreviatedWeekdayName(d.getDay(), localizer); },
// "D":"" not used
// "F":"" not used
// "w":"" not used
// "W":"" not used
    "a1":function(d, localizer) { return (d.getHours() < 12) ? Date.ampmName(0, localizer) : Date.ampmName(1, localizer); }
// "k":"" not used
// "K":"" not used
// "Z":"" not used
};

Date.prototype.stringWithContextualOrICUDateFormat = function(fullFormat, contextualFormat, localizer)
{
    // "ContextualFormat" is an ICU format string with the extension that CCCCC represents the
    // contextual date, "Yesterday" or "Today".  
    
    var result;
    var now = new Date();

    // Calculate the beginning of today in local time (not GMT/UTC).
    var beginningOfToday = new Date(now.getFullYear(), now.getMonth(), now.getDate());
    var beginningOfYesterday = new Date(Number(beginningOfToday.getTime()) - 24 * 60 * 60 * 1000);
    var beginningOfTomorrow = new Date(Number(beginningOfToday.getTime()) + 24 * 60 * 60 * 1000);

    var contextualDate;
    if ((this.getTime() >= beginningOfToday.getTime()) && (this.getTime() < beginningOfTomorrow.getTime()))
    {
        contextualDate = localizer.localizedString("Today");
    }
    else if ((this.getTime() >= beginningOfYesterday.getTime()) && (this.getTime() < beginningOfToday.getTime()))
    {
        contextualDate = localizer.localizedString("Yesterday");
    }

    if (contextualDate)
    {
        result = this.stringWithICUDateFormat(contextualFormat, localizer);
        result = result.replace(/C+/g, contextualDate);
    }
    else
    {
        result = this.stringWithICUDateFormat(fullFormat, localizer);
    }

    return result;
}

// Returns a string representation of the date using the date format string and localizer provided.
// Not all ICU formats are supported.
//
Date.prototype.stringWithICUDateFormat = function(format, localizer)
{
    var index = 0;
    var outFormat = "";
    var formatCode = "";
    var formatCount = 0;
    var processingText = false;
    
    while (true)
    {
        var ch;
        if (index >= format.length)
        {
            ch = "";
        }
        else
        {
            ch = format.charAt(index++);
        }
        
        if (ch != "" && ch == formatCode)
        {
            formatCount++;
        }
        else
        {
            if (formatCode.length > 0)
            {
                // Finish with old code
                var formatKey = formatCode + String(formatCount);
                try
                {
                    if (formatConversion[formatKey] != undefined) {
                        outFormat += formatConversion[formatKey](this, localizer);
                    }
                }
                catch (e)
                {
                    print(e);
                    return "";
                }
                formatCode = "";
                formatCount = 0;
            }
            if (ch == "") break;
            
            if (processingText)
            {
                if (ch == "'")
                {
                    processingText = false;
                }
                else
                {
                    if (ch == "\"")
                    {
                        ch == "'"
                    }
                    outFormat += ch;
                }
                continue;
            }
            
            if ("GyMdhHmsSEDFwWakKZ".indexOf(ch) >= 0)
            {
                formatCode = ch;
                formatCount = 1;
            }
            else if (ch == "'")
            {
                processingText = true;
            }
            else
            {
                outFormat += ch;
            }
        }
    }
    return outFormat;
}


//>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
