//
// iWeb - iWebMediaGrid.js
// Copyright 2007-2008 Apple Inc.
// All rights reserved.
//


// @import iWebSite.js
// @import iWebImage.js


//<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<

//  Subclasses of iWFeed must implement p_interpretItems(baseURL, items)
//
//  Responsibility: mmurrett
//  Reviewers: mmurrett, krevis, ksmyth
//


var IWAllFeeds = {};

function IWCreateFeed(url)
{
    var feed = IWAllFeeds[url];

    if(feed == null)
    {
        feed = new IWFeed(url);
    }

    return feed;
}


var IWFeed = Class.create({
    initialize: function(url)
    {
        if(url)
        {
            if(IWAllFeeds.hasOwnProperty(url))
            {
                iWLog("warning -- use IWCreateFeed rather than new IWFeed and you'll get better performance");
            }
    
            this.mURL = url;
            this.mLoading = false;
            this.mLoaded = false;
            this.mCallbacks = [];
    
            this.mImageStream = null;
    
            IWAllFeeds[url] = this;
        }
    },

    // returns this feed's URL
    sourceURL: function()
    {
        return this.mURL;
    },

    // causes this feed to load
    //
    // arguments:
    //  callback        (optional) a function to call when the feed feed loads.  An image stream
    //                  is supplied as the only argument to this function.
    load: function(baseURL, callback)
    {
        if(this.mLoaded && (callback != null))
        {
            callback(this.mImageStream);
        }
        else
        {
            if(callback != null)
            {
                this.mCallbacks.push(callback);
            }

            if(this.mLoading == false)
            {
                this.mLoading = true;
                this.p_sendRequest(baseURL);
            }
        }
    },

    p_sendRequest: function(baseURL)
    {
        var url = this.mURL.toRelativeURL(baseURL);
        new Ajax.Request(url, {
            method: 'get',
            onSuccess: this.p_onload.bind(this, baseURL),
            onFailure: this.p_requestFailed.bind(this, baseURL)
        });
    },

    p_requestFailed: function(baseURL, req)
    {
        iWLog("There was a problem (" + req.status + ") retrieving the feed:\n\r" + req.statusText);

        if(req.status == 500)
        {
            iWLog("working around status 500 by trying again...");
            window.setTimeout(this.p_sendRequest.bind(this, baseURL), 100);
        }
    },

    p_onload: function(baseURL, req)
    {
        var collectionItem;
        var doc = ajaxGetDocumentElement(req);
        var items = $A(doc.getElementsByTagName('item'));
        this.mImageStream = this.p_interpretItems(baseURL, items);

        // Call everyone back who is waiting for the load to complete.
        this.p_postLoadCallbacks(this.mImageStream);
    },

    p_postLoadCallbacks: function(imageStream)
    {
        for(var i = 0; i < this.mCallbacks.length; ++i)
        {
            this.mCallbacks[i](imageStream);
        }

        this.mLoaded = true;
    },

    p_applyEntryOrder: function(imageStream, entryGUIDs)
    {
        var orderedStream = [];

        // In order to avoid an O(n^2) sort, build an associate array
        // mapping from imageStream entry guids to the corresponding index.
        var guidToIndex = [];
        for(var i=0; i<imageStream.length; i++)
        {
            var streamEntryGUID = imageStream[i].guid();
            if(streamEntryGUID)
            {
                guidToIndex[streamEntryGUID] = i;
            }
        }
    
        // And use this mapping to generate the sorted array in O(n) time.
        for(var i=0; i<entryGUIDs.length; i++)
        {
            var index = guidToIndex[entryGUIDs[i]];
            if(index !== undefined)
            {
                orderedStream.push(imageStream[index]);
            }
        }

        (function(){return orderedStream.length == entryGUIDs.length}).assert();
    
        return orderedStream;
    },

    p_firstElementByTagNameNS: function(element, ns, tag)
    {
        var child = null;

        for (child = element.firstChild; child != null; child = child.nextSibling)
        {
            if (child.baseName == tag || // IE
                child.localName == tag)  // Safari and Firefox
            {
                if (ns == null || ns == "" || child.namespaceURI == ns)
                {
                    break;
                }
            }
        }

        return child;
    }
});
//>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>


//<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<

//  Responsibility: mmurrett
//  Reviewers: mmurrett, krevis, ksmyth
//


var IWStreamEntry = Class.create({
    initialize: function(thumbnailURL, title, richTitle, guid)
    {
        if(arguments.length > 0)
        {
            if(thumbnailURL)
            {
                this.mThumbnail = IWCreateImage(thumbnailURL);
            }

            if(title)
            {
                this.mTitle = title.stringByEscapingXML().stringByConvertingNewlinesToBreakTags();
            }
            if(richTitle)
            {
                this.mRichTitle = richTitle;
            }
            // NOTE: both mTitle and richTitle should now contain xhtml markup.
        
            if(guid)
            {
                this.mGUID = guid;
            }
        }
    },

    setThumbnailURL: function(thumbnailURL)
    {    
        this.mThumbnail = IWCreateImage(thumbnailURL);
    },

    loadThumbnail: function(callback)
    {
        this.thumbnail().load(callback);
    },

    unloadThumbnail: function()
    {
        this.thumbnail().unload();
    },

    thumbnailNaturalSize: function()
    {
        return this.thumbnail().naturalSize();
    },

    thumbnail: function()
    {
        return this.mThumbnail;
    },

    micro: function()
    {
        return this.thumbnail();
    },

    mipThumbnail: function()
    {
        return this.thumbnail();
    },

    title: function()
    {
        return this.mTitle;
    },

    richTitle: function()
    {
        return this.mRichTitle ? this.mRichTitle : this.mTitle;
    },

    metric: function()
    {
        return null;
    },

    guid: function()
    {
        return this.mGUID;
    },

    isMovie: function()
    {
        return false;
    },

    commentGUID: function()
    {
        return null;
    },

    showCommentIndicator: function()
    {
        return true;
    },

    badgeMarkupForRect: function(rect)
    {
        return IWStreamEntryBadgeMarkup(rect, this.isMovie(), this.showCommentIndicator() ? this.commentGUID() : null);
    },

    thumbnailMarkupForRect: function(rect)
    {
        return imgMarkup(this.thumbnail().sourceURL(), rect.position(), "", this.mTitle) +
                this.badgeMarkupForRect(rect);
    },

    didInsertThumbnailMarkupIntoDocument: function()
    {
    // no-op
    }
});



function IWStreamEntryBadgeMarkup(rect, isMovie, commentGUID)
{
    var kBadgeWidth = 16.0;
    var kBadgeHeight = 16.0;

    var badgeRect = new IWRect(rect.origin.x, rect.maxY() - kBadgeHeight, rect.size.width, kBadgeHeight);

    var markup = "";

    if(isMovie)
    {
        markup += '<div style="background-color: black; ' + badgeRect.position() + iWOpacity(0.75) + '"></div>';
    }

    var badgeSize = new IWSize(kBadgeWidth, kBadgeHeight);

    if(isMovie)
    {
        var movieImage = IWImageNamed("movie overlay");
        var movieRect = new IWRect(badgeRect.origin, badgeSize);

        markup += imgMarkup(movieImage.sourceURL(), movieRect.position(), 'class="badge-overlay"');
    }

    var commentLocation = new IWPoint(badgeRect.maxX() - kBadgeWidth, badgeRect.origin.y);

    if(commentGUID)
    {
        var commentImage = IWImageNamed("comment overlay");
        var commentRect = new IWRect(commentLocation, badgeSize);

        markup += imgMarkup(commentImage.sourceURL(), commentRect.position() + "display: none; ", 'class="badge-overlay" id="comment-badge-' + commentGUID + '"');
    }

    return markup;
}
//>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>


//<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<

var IWCommentableStreamEntry = Class.create(IWStreamEntry, {
    initialize: function($super, assetURL, showCommentIndicator, thumbnailURL, title, richTitle, guid)
    {
        $super(thumbnailURL, title, richTitle, guid);
        this.mAssetURL = assetURL;
        this.mShowCommentIndicator = showCommentIndicator;
    },

    commentGUID: function()
    {
        return this.guid();
    },

    showCommentIndicator: function()
    {
        return this.mShowCommentIndicator;
    },

    didInsertThumbnailMarkupIntoDocument: function()
    {
        if(this.mAssetURL && hostedOnDM())
        {
            IWCommentCountForURL(this.mAssetURL, this.p_commentCountCallback.bind(this));
        }
        else
        {
            this.p_commentCountCallback(0);
        }
    },

    commentAssetURL: function()
    {
        return this.mAssetURL;
    },

    p_commentCountCallback: function(commentCount)
    {
        this.mCommentCount = commentCount;
        if(this.mCommentCount > 0)
        {
            $('comment-badge-' + this.commentGUID()).show();
        }
    }
});
//>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>


//<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<


var IWImageStreamEntry = Class.create(IWCommentableStreamEntry, {
    initialize: function($super, assetURL, showCommentIndicator, imageURL, thumbnailURL, microURL, mipThumbnailURL, title, richTitle, guid)
    {
        $super(assetURL, showCommentIndicator, thumbnailURL, title, richTitle, guid);
        this.mImage = IWCreateImage(imageURL);
        if(microURL)
        {
            this.mMicro = IWCreateImage(microURL);
        }
        if(mipThumbnailURL)
        {
            this.mMIPThumbnail = IWCreateImage(mipThumbnailURL);
        }
    },

    setImageURL: function(imageURL)
    {
        this.mImage = IWCreateImage(imageURL);
    },

    image: function()
    {
        return this.mImage;
    },

    micro: function()
    {
        return this.mMicro ? this.mMicro : this.thumbnail();
    },

    mipThumbnail: function()
    {
        return this.mMIPThumbnail ? this.mMIPThumbnail : this.thumbnail();
    },

    targetURL: function()
    {
        return this.mImage.sourceURL();
    },

    slideshowValue: function(imageType)
    {
        var image = this[imageType]();
        return {image: image, caption: this.title()};
    }
});
//>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>


//<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<

var IWMovieStreamEntry = Class.create(IWCommentableStreamEntry, {
    initialize: function($super, assetURL, showCommentIndicator, movieURL, thumbnailURL, title, richTitle, movieParams, guid)
    {
        $super(assetURL, showCommentIndicator, thumbnailURL, title, richTitle, guid);
        this.mMovieURL = movieURL;
        this.mMovieParams = movieParams;
    },

    movieURL: function()
    {
        return this.mMovieURL;
    },

    targetURL: function()
    {
        return this.movieURL();
    },

    isMovie: function()
    {
        return true;
    },

    setMovieParams: function(params)
    {
        this.mMovieParams = params;
    },

    slideshowValue: function(imageType)
    {
        return {image: this.thumbnail(),
                movieURL: this.movieURL(),
                caption: this.title(),
                params: this.mMovieParams};
    }
});
//>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>


//<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<

var IWMediaStreamPageEntry = Class.create(IWStreamEntry, {
    initialize: function($super, targetPageURL, thumbnailURL, title, richTitle, guid)
    {
        if(arguments.length > 0)
        {
            $super(thumbnailURL, title, richTitle, guid);
            this.mTargetPageURL = targetPageURL;
        }
    },

    thumbnailNaturalSize: function()
    {
        return new IWSize(4000, 3000);
    },

    targetURL: function()
    {
        return this.mTargetPageURL;
    },

    positionedThumbnailMarkupForRect: function(rect)
    {
        return IWMediaStreamPageEntryPositionedThumbnailMarkupForRect(this.mThumbnail, rect);
    }
});

function IWMediaStreamPageEntryPositionedThumbnailMarkupForRect(thumbnail, rect)
{
    var thumbnailSize = thumbnail.naturalSize();
    var scale = Math.max(rect.size.width / thumbnailSize.width,
                         rect.size.height / thumbnailSize.height);

    var imageSize = thumbnailSize.scale(scale, scale, true);
    var imagePosition = new IWPoint((rect.size.width - imageSize.width) / 2,
                                    (rect.size.height - imageSize.height) / 2);
    imagePosition = imagePosition.scale(1, 1, true);    // just round

    var imageRect = new IWRect(imagePosition, imageSize);

    return imgMarkup(thumbnail.sourceURL(), imageRect.position());
}
//>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>


//<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<

var IWMediaStreamPhotoPageEntryPrefs = {};
var IWMediaStreamPhotoPageEntries = {};

function IWMediaStreamPhotoPageSetPrefs(slideshowPrefs)
{
    IWMediaStreamPhotoPageEntryPrefs = slideshowPrefs;
}

var IWMediaStreamPhotoPageEntryUniqueId = 0;

var IWMediaStreamPhotoPageEntry = Class.create(IWMediaStreamPageEntry, {
    initialize: function($super, streamScriptURL, targetPageURL, title, contentsFunction, guid)
    {
        $super(targetPageURL, null, title, null, guid);
        this.mStreamScriptURL = streamScriptURL;
        this.mContentsFunction = contentsFunction;
    },

    loadThumbnail: function(callback)
    {
        this.mThumbnailCallback = callback;

        this.mSlideshowId = 'gridEntry' + IWMediaStreamPhotoPageEntryUniqueId++;
        IWMediaStreamPhotoPageEntries[this.mSlideshowId] = this;

        var loadingArea = IWCreateLoadingArea();
    
        // The iframe id must be origional to avoid a Safari bug where the iframe src is not updated in the DOM on a pre-existing iframe
        // Append two IDs so unique Time(ms) OR unique MediaStreamId guarantees unique iFrame
        var uniqueId = 'iFrame_' + new Date().getTime() + IWMediaStreamPhotoPageEntryUniqueId;
        loadingArea.innerHTML = '<iframe id=' + uniqueId + ' src="streamloader.html?scriptURL=' + this.mStreamScriptURL + '&id=' + this.mSlideshowId + '" style="position: absolute; visibility: hidden; ">' +
            '</iframe>';
    },

    streamDidLoad: function(media)
    {
        this.mMedia = media;

        if(this.mMedia && this.mMedia.length > 0)
        {
            this.mThumbnail = this.mMedia[0].mipThumbnail();
            this.mThumbnail.load(this.mThumbnailCallback);
        }
        else
        {
            this.mThumbnailCallback();
        }
    },

    metric: function()
    {
        var photoCount = 0;
        var clipCount = 0;

        if(this.mMedia)
        {
            for(var index = 0; index < this.mMedia.length; ++index)
            {
                if(this.mMedia[index].isMovie())
                    ++clipCount;
                 else
                    ++photoCount;
            }
        }

        return this.mContentsFunction(photoCount, clipCount);
    },

    thumbnailMarkupForRect: function(rect)
    {
        var markup = "";

        if(this.mThumbnail)
        {
            // create a div to house the widget, and an iframe to load the stub page that will call us back
            //  (at IWMediaStreamPhotoPageSetMediaStream) with the media stream.  The stub page also passes
            //  us the "id" argument that we put in its URL, so that we can tell the widget where to load.
            markup = '<div id="' + this.mSlideshowId + '" style="overflow: hidden; ' + rect.position() + '" onclick="window.location.href = \'' + this.targetURL() + '\'">' +
                         this.positionedThumbnailMarkupForRect(rect) +
                         '<div id="' + this.mSlideshowId + '-slideshow_placeholder" style="position: absolute; left: 0px; top: 0px; height: 100%; width: 100%; overflow: hidden; ">' +
                         '</div>' +
                     '</div>';
        }

        return markup;
    },

    didInsertThumbnailMarkupIntoDocument: function()
    {
        if(this.mThumbnail)
        {
            if(isiPhone == false)
            {
                var prefs = IWMediaStreamPhotoPageEntryPrefs;
                prefs["mediaStreamObject"] = {load: function(media, baseURL, callback){ callback(media); }.bind(null, this.mMedia)};
        
                new SlideshowGlue(this.mSlideshowId,
                                    '../Scripts/Widgets/Slideshow',
                                    '../Scripts/Widgets/SharedResources',
                                    '..',
                                    prefs);
            }
        }
    }
});

function IWMediaStreamPhotoPageSetMediaStream(mediaStream, slideshowId)
{
    mediaStream.load(IWMediaStreamPhotoPageEntryPrefs.baseURL, function(slideshowId, media)
    {
        IWMediaStreamPhotoPageEntries[slideshowId].streamDidLoad(media);
    }.bind(null, slideshowId));
}
//>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>


//<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<

var IWMediaStreamMediaPageEntryUniqueId = 0;

var IWMediaStreamMediaPageEntry = Class.create(IWMediaStreamPageEntry, {
    initialize: function($super, targetPageURL, thumbnailURL, title, contents, isMovie, guid)
    {
        $super(targetPageURL, thumbnailURL, title, null, guid);
        this.mContents = contents;
        this.mIsMovie = isMovie;
    },

    isMovie: function()
    {
        return this.mIsMovie;
    },

    metric: function()
    {
        return this.mContents;
    },

    thumbnailMarkupForRect: function(rect)
    {
        // <rdar://problem/5058009> -- need to compensate for the rect origin when generating the
        // badge markup because in this case the badge is positioned within a div that already
        // includes the rect position.
        var badgeRect = new IWRect(new IWPoint(0, 0), rect.size);
    
        var thumbnailMarkup = this.positionedThumbnailMarkupForRect(rect) + this.badgeMarkupForRect(badgeRect);
        var idAttribute = "";
        var playButtonMarkup = "";

        if(this.isMovie())
        {
            this.mPlayButtonId = 'movieEntry' + IWMediaStreamMediaPageEntryUniqueId++;

            idAttribute = 'id="' + this.mPlayButtonId + '"';
            playButtonMarkup =  '<div id="' + this.mPlayButtonId + '-play_button" class="play_button">' +
                                    '<div>' +
                                    '</div>' +
                                '</div>';
        }

        var markup =    '<div ' + idAttribute + ' style="overflow: hidden; ' + rect.position() + '" onclick="window.location.href = \'' + this.targetURL() + '\'">' +
                            thumbnailMarkup +
                            playButtonMarkup +
                        '</div>';

        return markup;
    },

    didInsertThumbnailMarkupIntoDocument: function()
    {
        if(this.isMovie())
        {
            if(isiPhone == false)
            {
                new PlayButton(this.mPlayButtonId,
                                '../Scripts/Widgets/PlayButton',
                                '../Scripts/Widgets/SharedResources',
                                '..',
                                {});
            }
        }
    }
});
//>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>


//<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<

var gPhotoFormats = [];
var gClipFormats = [];


function IWCreateMediaCollection(url, slideshowEnabled, transitionIndex, photoFormats, clipFormats)
{
    var feed = IWAllFeeds[url];

    if(feed == null)
    {
        if(gPhotoFormats.length == 0)
        {
            gPhotoFormats = photoFormats;
        }
        if(gClipFormats.length == 0)
        {
            gClipFormats = clipFormats;
        }

        feed = new IWMediaCollection(url, slideshowEnabled, transitionIndex);
    }

    return feed;
}


var IWMediaCollection = Class.create(IWFeed, {
    initialize: function($super, url, slideshowEnabled, transitionIndex)
    {
        $super(url);
        this.mSlideshowEnabled = slideshowEnabled;
        this.mTransitionIndex = transitionIndex;
    },

    p_interpretItems: function(baseURL, items)
    {
        var iWebNamespace = 'http://www.apple.com/iweb';
        var thumbnailNamespace = 'urn:iphoto:property';

        var truthFeed = this.mURL.indexOf('?webdav-method=truthget') != -1;

        if(truthFeed)
        {
            iWebNamespace = thumbnailNamespace = "urn:iweb:";
        }

        if(IWMediaStreamPhotoPageSetPrefs)
        {
            IWMediaStreamPhotoPageSetPrefs({slideshowEnabled: this.mSlideshowEnabled,
                                            fadeIn: true,
                                            showOnMouseOver: true,
                                            photoDuration: 2,
                                            startIndex: 1,
                                            scaleMode: "fill",
                                            transitionIndex: this.mTransitionIndex,
                                            imageType: "mipThumbnail",
                                            movieMode: kPosterFrameOnly,
                                            baseURL: baseURL});
        }

        // extract images and thumbnails from the feed
        var mediaStream = [];
        for(var i = 0; i < items.length; ++i)
        {
            var item = items[i];

            var link = null;
            if(truthFeed)
            {
                link = this.p_firstElementByTagNameNS(item, iWebNamespace, 'link');
            }

            if(link == null)
            {
                link = item.getElementsByTagName('link')[0];
            }

            if(link != null)
            {
            // TODO (mm) -- to much of this is identical to iWPhotocast.js
                var titleText = '';
                var title = item.getElementsByTagName('title')[0];
                if(title != null)
                {
                    titleText = title.firstChild.nodeValue;
                }

                var skip = false;
                if(truthFeed)
                {
                    // re-get the title
                    title = this.p_firstElementByTagNameNS(item, iWebNamespace, 'title');
                    if(title != null)
                    {
                        titleText = title.firstChild.nodeValue;
                    }
                    else
                    {
                        // this is some bogus entry, skip it
                        skip = true;
                    }
                }

                if(skip == false)
                {
                    var pageURL = link.firstChild.nodeValue.toRelativeURL(baseURL);
    
                    var entry = null;
    
                    var guidNode = item.getElementsByTagName('useritemguid')[0];
                    var guid = null;
                    if(guidNode != null)
                    {
                        guid = guidNode.firstChild.nodeValue;
                    }
    
                    var thumbnail = this.p_firstElementByTagNameNS(item, thumbnailNamespace, 'thumbnail') ||
                                    item.getElementsByTagName('thumbnail')[0];
                    if(thumbnail)
                    {
                        var thumbnailURL = transparentGifURL();
                        if(thumbnail.firstChild && thumbnail.firstChild.nodeValue)
                        {
                            thumbnailURL = thumbnail.firstChild.nodeValue.toRelativeURL(baseURL);
                        }

                        var contentsText = null;
                        var contents = this.p_firstElementByTagNameNS(item, iWebNamespace, 'contents');
                        if(contents)
                        {
                            contentsText = contents.firstChild.nodeValue;
                        }
                        var isMovie = false;
                        var isMovieValue = this.p_firstElementByTagNameNS(item, iWebNamespace, 'is-movie');
                        if(isMovieValue && isMovieValue.firstChild && isMovieValue.firstChild.nodeValue)
                        {
                            isMovie = (isMovieValue.firstChild.nodeValue == 'true');
                        }

                        entry = new IWMediaStreamMediaPageEntry(pageURL, thumbnailURL, titleText, contentsText, isMovie, guid);
                    }
                    else
                    {
                        var pageName = pageURL.lastPathComponent().stringByDeletingPathExtension();
                        var pageScriptURL = pageURL.stringByDeletingLastPathComponent().stringByAppendingPathComponent(pageName + "_files").stringByAppendingPathComponent(pageName + ".js");
                        entry = new IWMediaStreamPhotoPageEntry(pageScriptURL, pageURL, titleText, albumContentsFunction, guid);
                    }
    
                    mediaStream.push(entry);
                }
            }
        }

        return mediaStream;
    }
});

function albumContentsFunction(photoCount, clipCount)
{
// IMPORTANT: if you make any changes to this function, make sure to make corresponding changes to
//              +[BLMediaStreamMediaStreamEntryInfo p_contentsStringForPhotoCount:clipCount:]
    var contents = "";

    photoFormat = gPhotoFormats[Math.min(photoCount, 2)];
    clipFormat = gClipFormats[Math.min(clipCount, 2)];

    photoFormat = photoFormat.replace(/%d/, photoCount);
    clipFormat = clipFormat.replace(/%d/, clipCount);

    if(clipFormat && clipFormat.length > 0 && photoCount == 0)
    {
        contents = clipFormat;
    }
    else
    {
        contents = photoFormat;
        if(clipFormat && clipFormat.length > 0)
        {
            contents += ", " + clipFormat;
        }
    }

    return contents;
}
//>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>


//<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<

function IWCreatePhotocast(url, showCommentIndicator)
{
    var feed = IWAllFeeds[url];

    if(feed == null)
    {
        feed = new IWPhotocast(url, showCommentIndicator);
    }

    return feed;
}


var IWPhotocast = Class.create(IWFeed, {
    initialize: function($super, url, showCommentIndicator)
    {
        $super(url);
        this.mShowCommentIndicator = showCommentIndicator;
    },

    p_interpretItems: function(baseURL, items)
    {
        // extract images and thumbnails from the feed
        var imageStream = [];
        for(var i = 0; i < items.length; ++i)
        {        
            var item = items[i];
            enclosure = item.getElementsByTagName('enclosure')[0];
            if(enclosure != null)
            {            
                var titleText = '';
                var title = item.getElementsByTagName('title')[0];
                if(title && title.firstChild && title.firstChild.nodeValue)
                {
                    titleText = title.firstChild.nodeValue;
                }

                var iWebNamespace = 'http://www.apple.com/iweb';
                var thumbnailNamespace = 'urn:iphoto:property';

                var richTitleHTML;
                var richTitle = this.p_firstElementByTagNameNS(item, iWebNamespace, 'richTitle');
                if(richTitle && richTitle.firstChild && richTitle.firstChild.nodeValue)
                {
                    richTitleHTML = richTitle.firstChild.nodeValue.stringByUnescapingXML();
                }

                var guidNode = item.getElementsByTagName('guid')[0];
                var guid = null;
                if(guidNode != null)
                {
                    guid = guidNode.firstChild.nodeValue;
                }
            
                var thumbnail = this.p_firstElementByTagNameNS(item, thumbnailNamespace, 'thumbnail') ||
                                item.getElementsByTagName('thumbnail')[0]; // Just in case the thumbnail namespace is incorrect, try to find any thumb, regardless of namespace.

                // Assert that the thumbnail is not null... but attempt to work around if it is.  Right now there's no point to working around like
                // that b/c the assert will throw an exception, but we will (I believe) be removing asserts before we ship and we should try
                // to show the rest of the page as well as we can.
                IWAssert(function(){return thumbnail != null}, "Could not get thumbnail from feed.  Server configuration may have changed.");

                if(thumbnail)
                {
                    var entry = null;
                    var type = enclosure.getAttribute("type").split("/");
                    var thumbnailURL = thumbnail.firstChild.nodeValue.toRebasedURL(baseURL);
                    if(type[0] == "video")
                    {
                        var movieURL = enclosure.getAttribute('url').toRebasedURL(baseURL);
                        var movieParams = null;
                        var movieParamsElement = this.p_firstElementByTagNameNS(item, iWebNamespace, 'movieParams');
                        if(movieParamsElement && movieParamsElement.firstChild && movieParamsElement.firstChild.nodeValue)
                        {
                            var movieParamsJSON = movieParamsElement.firstChild.nodeValue.stringByUnescapingXML();
                            movieParams = eval('(' + movieParamsJSON + ')');
                        }
            
                        var assetURL = movieURL.stringByDeletingLastPathComponent();
                        IWAssert(function(){return assetURL != null}, "could not determine asset URL for movie: "+movieURL);
                        entry = new IWMovieStreamEntry(assetURL, this.mShowCommentIndicator, movieURL, thumbnailURL, titleText, richTitleHTML, movieParams, guid);
                    }
                    else
                    {
                        var imageURL = enclosure.getAttribute('url').toRebasedURL(baseURL);
                        var assetURL = imageURL.urlStringByDeletingQueryAndFragment().stringByDeletingPathExtension();

                        var microURL = null;
                        var micro = this.p_firstElementByTagNameNS(item, iWebNamespace, 'micro');
                        if(micro && micro.firstChild && micro.firstChild.nodeValue)
                        {
                            microURL = micro.firstChild.nodeValue.toRebasedURL(baseURL);
                        }
                        if(!microURL)
                        {
                            microURL = (imageURL.urlStringByDeletingQueryAndFragment() + "?derivative=micro").toRebasedURL(baseURL);
                        }

                        var mipThumbnailURL = null;
                        var mipThumbnail = this.p_firstElementByTagNameNS(item, iWebNamespace, 'mip-thumbnail');
                        if(mipThumbnail && mipThumbnail.firstChild && mipThumbnail.firstChild.nodeValue)
                        {
                            mipThumbnailURL = mipThumbnail.firstChild.nodeValue.toRebasedURL(baseURL);
                        }
                        if(!mipThumbnailURL)
                        {
                            mipThumbnailURL = (imageURL.urlStringByDeletingQueryAndFragment() + "?derivative=mip&alternate=thumb").toRebasedURL(baseURL);
                        }

                        IWAssert(function(){return assetURL != null}, "could not determine asset URL for image: "+imageURL);
                        entry = new IWImageStreamEntry(assetURL, this.mShowCommentIndicator, imageURL, thumbnailURL, microURL, mipThumbnailURL, titleText, richTitleHTML, guid);
                    }
                    imageStream.push(entry);
                }
            }
        }

        return imageStream;
    }
});
//>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>


//<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<

var kPhotoViewMovieControllerHeight = 16;

var kShowMovie = 0;
var kAutoplayMovie = 1;
var kPosterFrameOnly = 2;


function setFrameOptionallyMovingContents(element, frame, moveContents)
{
    var oldOrigin = new IWPoint(element.offsetLeft, element.offsetTop);

    $(element).setStyle({
        left: px(frame.origin.x),
        top: px(frame.origin.y),
        width: px(frame.size.width),
        height: px(frame.size.height)
    });

    var offset = new IWPoint(oldOrigin.x - frame.origin.x, oldOrigin.y - frame.origin.y);
    
    if(moveContents)
    {
        offsetChildren(element, offset, false);
    }
    
    return offset;
}

function offsetChildren(element, offset, reverse)
{
    for(var node = element.firstChild; node; node = node.nextSibling)
    {
        var left = parseFloat(node.style.left || 0);
        var top = parseFloat(node.style.top || 0);
        $(node).setStyle({
            left: px(reverse ? left - offset.x : left + offset.x),
            top: px(reverse ? top - offset.y : top + offset.y)
        });
    }
}

var PhotoViewWaitingForDonePlaying = [];
function PhotoViewDonePlaying(index)
{
    PhotoViewWaitingForDonePlaying[index]();
    return false;
}

var PhotoView = Class.create({
    initialize: function(container, scaleMode, reflectionHeight, reflectionOffset, backgroundColor, movieMode, captionHeight)
    {
        this.scaleMode = "fit";
        this.reflectionHeight = 0;
        this.reflectionOffset = 0;
        this.backgroundColor = "black";
        this.movieMode = kShowMovie;
        this.captionHeight = 0;

        if(scaleMode)
        {
            this.scaleMode = scaleMode;
        }

        if(reflectionHeight)
        {
            this.reflectionHeight = reflectionHeight;
            if(reflectionOffset)
            {
                this.reflectionOffset = reflectionOffset;
            }

            if(windowsInternetExplorer)
            {
                this.reflection = document.createElement("img");
            }
            else
            {
                this.reflection = document.createElement("canvas");
            }
        }

        if(captionHeight)
        {
            this.captionHeight = captionHeight;
            this.caption = document.createElement("div");
        }

        if(backgroundColor)
        {
            this.backgroundColor = backgroundColor;
        }

        if(movieMode !== undefined)
        {
            this.movieMode = movieMode;
        }

        var div = document.createElement("div");

        var canvas = document.createElement("canvas");
        if(canvas.getContext && !isOpera)
        {
            div.appendChild(canvas);
            this.canvas = canvas;
        }
        else
        {
            var img = document.createElement("img");
            img.src = transparentGifURL();
            div.appendChild(img);
            this.img = img;
        }

        if(this.reflection)
        {
            div.appendChild(this.reflection);
        }

        if(this.caption)
        {
            div.appendChild(this.caption);
        }

        container.appendChild(div);

        this.box = container;
        this.div = div;

        this.initStyle();
    },

    boxSize: function()
    {
        return {width: this.box.offsetWidth, height: this.box.offsetHeight};
    },

    initStyle: function ()
    {
        var box_size = this.boxSize();

        $(this.div).setStyle({
            position: "absolute",
            width: px(box_size.width),
            height: px(box_size.height),
            backgroundColor: this.backgroundColor
        });

        if(this.canvas)
        {
            $(this.canvas).setStyle({position: "absolute", left: 0, top: 0});
            $(this.canvas).setAttribute("width", box_size.width);
            $(this.canvas).setAttribute("height", box_size.height - Math.max(this.reflectionHeight + this.reflectionOffset, this.captionHeight));
        
            if(!windowsInternetExplorer)
            {
                this.canvas.style.zIndex = "inherit"; // <rdar://problem/5053013>
            }
        }
        else if(this.img)
        {
            this.img.style.position = "absolute";
        }

        if(this.reflection)
        {
            this.reflection.style.position = "absolute";

            if(!windowsInternetExplorer)
            {
                this.reflection.setAttribute("width", box_size.width);
                this.reflection.setAttribute("height", this.reflectionHeight);
            }
        }

        if(this.caption)
        {
            $(this.caption).setStyle({
                position: "absolute",
                top: px(box_size.height - this.captionHeight * 0.6),
                left: 0,
                width: px(box_size.width)
            });
            this.caption.className = "caption";
        }

        this.resetSizeAndPosition();
    },

    setImage: function (photo)
    {
        if(this.photo !== photo)
        {
            if(this.photo)
            {
                this.photo.image.allowUnloading();
                this.photo.image.unload();
            }
            if(photo)
            {
                photo.image.preventUnloading();
            }
        }

        this.photo = photo;
        this.image = photo.image;
        this.movieURL = null;

        if(this.movieMode != kPosterFrameOnly && photo.movieURL)
        {
            this.movieURL = photo.movieURL;
        }

        if(this.caption)
        {
            this.caption.innerHTML = photo.caption || '';
        }

        var box_size = this.boxSize();
        box_size.height -= Math.max(this.reflectionHeight + this.reflectionOffset, this.captionHeight);

        if(this.movieURL && box_size.height > kPhotoViewMovieControllerHeight)
        {
            // leave space for the movie controller
            box_size.height -= kPhotoViewMovieControllerHeight;
        }

        var box_aspect = box_size.height / box_size.width;
        var img_size = this.image.naturalSize();
        var img_aspect = img_size.height / img_size.width;

        var w = box_size.width;
        var h = box_size.height;

        if ((img_aspect > box_aspect && this.scaleMode == "fit") ||
            (img_aspect < box_aspect && this.scaleMode == "fill"))
        {
            // Fit / Tall or Fill / Wide
            var ratio = box_size.height / img_size.height;
            if(this.scaleMode == "fit")
                ratio = Math.min(ratio, 1.0);
            w = Math.round(img_size.width * ratio);
            h = Math.round(img_size.height * ratio);
        }
        else if ((img_aspect < box_aspect && this.scaleMode == "fit") ||
                 (img_aspect > box_aspect && this.scaleMode == "fill"))
        {
            // Fit / Wide or Fill / Tall
            var ratio = box_size.width / img_size.width;
            if(this.scaleMode == "fit")
                ratio = Math.min(ratio, 1.0);
            w = Math.round(img_size.width * ratio);
            h = Math.round(img_size.height * ratio);
        }

        var x = Math.floor((box_size.width - w) / 2);
        var y = Math.floor((box_size.height - h) / 2);

        if(this.canvas)
        {
            if(this.scaleMode == "fit") // don't do this when filling, as it's not really correct
            {
                if(this.canvas.width != w || this.canvas.height != h)
                {
                    if(true)    // needed for Tiger Safari
                    {
                        var newCanvas = this.canvas.cloneNode(false);
                        newCanvas.setAttribute("width", w);
                        newCanvas.setAttribute("height", h);
                        this.canvas.parentNode.replaceChild(newCanvas, this.canvas);
                        this.canvas = newCanvas;
                    }
                    else
                    {
                        this.canvas.setAttribute("width", w);
                        this.canvas.setAttribute("height", h);
                    }
                    $(this.canvas).setStyle({top: px(y), left: px(x)});
                }
                if(w > 0 && h > 0)
                {
                    var context = this.canvas.getContext("2d");
                    context.clearRect(0, 0, w, h);
                    context.drawImage(this.image.imgObject(), 0, 0, w, h);
                }
            }
            else
            {
                if(w > 0 && h > 0)
                {
                    var context = this.canvas.getContext("2d");
                    context.clearRect(0, 0, this.canvas.width, this.canvas.height);
                    context.drawImage(this.image.imgObject(), x, y, w, h);
                }
            }
        }
        else if(this.img)
        {
            $(this.img).src = this.image.sourceURL();
            $(this.img).setStyle({width: px(w), height: px(h), top: px(y), left: px(x)});
        }

        this.updateReflection(x, y, w, h);

        if (windowsInternetExplorer)
        {
            // For some reason, immediately updating the reflection in IE doesn't
            // always take, the first call often ends up with the reflection being
            // blank.  However if we delay the call entirely, the old reflection
            // will stick around for a split second as you change images.  So, on
            // IE we call updateRefeltion twice; once inline (which may work, but
            // even if doesn't, it will at least clear out the old reflection), and
            // once slightly delayed which should look correct.
            setTimeout(this.updateReflection.bind(this, x, y, w, h), 0);
        }

        if(this.movieURL)
        {
            this.movieRect = new IWRect(x, y, w, h);
            this.p_updateMovieRect();
        }
    },

    willAnimate: function()
    {
        if(this.caption && this.caption.parentNode == this.div)
        {
            this.div.removeChild(this.caption);
        }
    },

    didAnimate: function()
    {
        if(this.caption && this.caption.parentNode != this.div)
        {
            this.div.appendChild(this.caption);
        }
    },

    transitionComplete: function()
    {
        if(this.movieURL && this.movieMode != kPosterFrameOnly)
        {
            this.p_showMovie();
        }
    },

    // based on reflection.js <http://cow.neondragon.net/stuff/reflection/>
    updateReflection: function(x, y, w, h)
    {
        if(this.reflection)
        {
            var opacity = 0.5;  // TODO (mm) -- don't hard-code this?

            this.reflection.style.top = px(y + h + this.reflectionOffset);

            if(windowsInternetExplorer)
            {
                this.reflection.src = this.image.sourceURL();

                $(this.reflection).setStyle({left: px(x), width: px(w), height: px(h)});
                this.reflection.style.filter = 'flipv' + ' progid:DXImageTransform.Microsoft.Alpha(opacity=' + opacity * 100 + ', style=1, finishOpacity=0, startx=0, starty=0, finishx=0, finishy=' + this.reflectionHeight / h * 100 + ')';
            }
            else if(this.reflection.getContext)
            {
                if(true)    // needed for Tiger Safari
                {
                    var newReflection = this.reflection.cloneNode(false);
                    newReflection.setAttribute("width", w);
                    this.reflection.parentNode.replaceChild(newReflection, this.reflection);
                    this.reflection = newReflection;
                }
                else
                {
                    this.reflection.setAttribute("width", w);
                }
                this.reflection.style.left = px(x);

                if(w > 0 && this.reflection.height > 0)
                {
                    var context = this.reflection.getContext("2d");

                    context.clearRect(0, 0, w, this.reflection.height);

                    context.save();
    
                    context.translate(0, h - 1);
                    context.scale(1, -1);
    
                    context.drawImage(this.image.imgObject(), 0, 0, w, h);
    
                    context.restore();
    
                    context.save();
                    var gradient = context.createLinearGradient(0, 0, 0, this.reflectionHeight);
    
                    gradient.addColorStop(1, this.backgroundColor);
                    var partiallyOpaque = "rgba(0,0,0," + opacity + ")";
    
                    var components = this.backgroundColor.toLowerCase().match(/#?([0-9a-f]{2})([0-9a-f]{2})([0-9a-f]{2})/);
                    if(components && components.length >= 4)
                    {
                        partiallyOpaque = "rgba(" + parseInt(components[1], 16) + ", " + parseInt(components[2], 16) + ", " + parseInt(components[3], 16) + ", " + opacity + ")";
                    }
                    gradient.addColorStop(0, partiallyOpaque);
    
                    context.fillStyle = gradient;
                    if (navigator.appVersion.indexOf('WebKit') != -1)
                    {
                        context.rect(0, 0, w, this.reflectionHeight * 2);
                        context.fill();
                    }
                    else
                    {
                        context.fillRect(0, 0, w, this.reflectionHeight * 2);
                    }
                    context.restore();
                }
            }
        }
    },

    updateSize: function ()
    {
        this.initStyle();
        if(this.photo)
        {
            this.photo.image.load(this.setImage.bind(this, this.photo));
        }
    },

    setZIndex: function (zIndex)
    {
        this.div.style.zIndex = zIndex;
    },

    upper: function ()
    {
        this.setZIndex(1);
    },

    lower: function ()
    {
        this.setZIndex(0);
    },

    show: function ()
    {
        this.div.style.visibility = "visible";
    },

    hide: function ()
    {
        this.p_destroyMovie();
        this.div.style.visibility = "hidden";
    },

    upperShown: function ()
    {
        this.upper();
        this.show();
    },

    upperHidden: function ()
    {
        this.hide();
        this.upper();
    },

    lowerShown: function ()
    {
        this.lower();
        this.show();
    },

    lowerHidden: function ()
    {
        this.hide();
        this.lower();
    },

    setClipPx: function (top, width, height, left)
    {
        this.div.style.clip = "rect(" + top + "px " + width + "px " + height + "px " + left + "px)";
    },

    setLeftPx: function (left)
    {
        this.div.style.left = px(left);
    },

    setTopPx: function (top)
    {
        this.div.style.top = px(top);
    },

    setClipToMax: function ()
    {
        var size = this.boxSize();
        this.div.style.clip = "rect(0px " + size.width + "px " + size.height + "px 0px)";
    },

    resetSizeAndPosition: function ()
    {
        $(this.div).setStyle({left: 0, top: 0});
        this.setClipToMax();
    },

    setOpacity: function (opacity)
    {
        IWSetDivOpacity(this.div, opacity)
    },

    p_appendParamToObject: function(name, value)
    {
        var param = document.createElement("param");
        param.name = name;
        param.value = value;
        this.object.appendChild(param);
    },

    p_stopMovie: function()
    {
        // stop the movie if it's still playing
        var movie = this.movieID ? document[this.movieID] : null;
        if(movie)
        {
            var status = null;
            try
            {
                status = movie.GetPluginStatus();
            }
            catch(e)
            {
                status = null;
            }
        
            if(status == null || status.startsWith("Error:") ||
                status == "Waiting" || status == "Loading" )
            {
                // error, we cannot stop the movie now
            }
            else
            {
                if(status == "Playable" || status == "Complete")
                {
                    if(movie.GetRate() > 0)
                    {
                        movie.Stop();
                    }
                }
            }
        }
    },

    p_destroyMovie: function()
    {
        if(this.movieID)
        {
            this.p_stopMovie();
            var movie = document[this.movieID];
            if(movie)
            {
                movie.style.display = "none";
            }
        
            this.div.removeChild(this.object);

            delete this.object;
            delete this.movieIndex;
            delete this.movieID;
        }
    },

    p_movieDidFinish: function()
    {
        NotificationCenter.postNotification(new IWNotification("PhotoViewMovieDidEnd", this, {}));
    },

    p_timeString: function(time)
    {
        var seconds = Math.floor(time);
        var frames = (time - seconds) * 30;
        var minutes = Math.floor(seconds / 60);
        var hours = Math.floor(minutes / 60);
        seconds -= minutes * 60;
        minutes -= hours * 60;

        if(minutes < 10)
        {
            minutes = "0" + minutes;
        }

        if(seconds < 10)
        {
            seconds = "0" + seconds;
        }

        frames = Math.round(frames * 10) / 10;
        if(frames < 10)
        {
            frames = "0" + frames;
        }

        var timeString = hours + ":" + minutes + ":" + seconds + ":" + frames;
        return timeString;
    },

    p_movieHeight: function()
    {
        var movieHeight = this.movieRect.size.height;
        if(this.photo.params.movieShowController)
        {
            movieHeight += kPhotoViewMovieControllerHeight;
        }

        return movieHeight;
    },

    p_updateMovieRect: function()
    {
        if(this.object)
        {
            this.object.setAttribute("width", this.movieRect.size.width);
            this.object.setAttribute("height", this.p_movieHeight());

            $(this.object).setStyle({
                position: "absolute",
                left: px(this.movieRect.origin.x),
                top: px(this.movieRect.origin.y)
            });
        }
    },

    p_showMovie: function()
    {
        this.object = document.createElement("object");

        if(this.object)
        {
            this.movieIndex = PhotoViewWaitingForDonePlaying.length;

            this.movieID = "photoViewMovie" + this.movieIndex;

            this.object.id = this.movieID;
  
            if(!windowsInternetExplorer)
            {
                this.object.type = "video/quicktime";
                this.object.setAttribute("data", this.movieURL);
                this.object.style.zIndex = "inherit";   // rdar://5131586
            }

            // Update the size and position of the movie
            this.p_updateMovieRect();

            var shouldAutoplay = this.movieMode == kAutoplayMovie || this.photo.params.movieAutoPlay;

            this.p_appendParamToObject("controller", this.photo.params.movieShowController ? "true" : "false");
            this.p_appendParamToObject("autoplay", shouldAutoplay ? "true" : "false");
            this.p_appendParamToObject("scale", "tofit");
            this.p_appendParamToObject("volume", "12800");      // TODO (mm) -- ??
            this.p_appendParamToObject("loop", this.photo.params.movieLoop != "SFDMovieNoLoop" ? "true" : "false");

            this.p_appendParamToObject("starttime", this.p_timeString(this.photo.params.movieStartTime * this.photo.params.movieDuration));
            this.p_appendParamToObject("endtime", this.p_timeString(this.photo.params.movieEndTime * this.photo.params.movieDuration));

            this.p_appendParamToObject("src", this.movieURL);

            PhotoViewWaitingForDonePlaying[this.movieIndex] = this.p_movieDidFinish.bind(this);

            this.p_appendParamToObject("qtnext1", "<javascript:PhotoViewDonePlaying(" + this.movieIndex + ");>");

            this.div.appendChild(this.object);
            if(windowsInternetExplorer)
            {
                this.object.classid="clsid:02BF25D5-8C17-4B23-BC80-D3488ABDDC6B";
                this.object.codebase="http://www.apple.com/qtactivex/qtplugin.cab";         
            }

    //        this.p_updateMovieParams();
        }
    },

    setMovieTime: function(time)
    {
        if(this.object)
        {
            var movie = document[this.movieID];
            movie.Stop();
            movie.SetTime(time * movie.GetDuration());
        }
    },

    p_updateMovieParams: function()
    {
        if(this.object)
        {
            var movie = document[this.movieID];
            if(movie)
            {
                var params = $H(this.photo.params);
                var self = this;
                params.each(function(pair)
                            {
                                if(pair.key == "movieShowController")
                                {
                                    if(movie.GetControllerVisible() != pair.value)
                                    {
                                        movie.SetControllerVisible(pair.value);
                                        self.object.setAttribute("height", self.p_movieHeight());
                                    }
                                }
                                else if(pair.key == "movieLoop")
                                {
                                    movie.SetIsLooping(pair.value != "SFDMovieNoLoop");
                                }
                                else if(pair.key == "movieAutoPlay")
                                {
                                    movie.SetAutoPlay(pair.value || this.movieMode == kAutoplayMovie);
                                }
                                else if(pair.key == "movieStartTime")
                                {
                                    movie.SetStartTime(pair.value * movie.GetDuration());
                                }
                                else if(pair.key == "movieEndTime")
                                {
                                    movie.SetEndTime(pair.value * movie.GetDuration());
                                }
                            });
            }
        }
    },

    setMovieParams: function(params)
    {
        var photo = this.photo;
        $H(params).each(function(pair)
                        {
                            photo.params[pair.key] = pair.value;
                        });
        this.p_updateMovieParams();
    }
});

/* ------------------- */

var SimpleAnimation = Class.create({
    initialize: function(oncomplete)
    {
        this.duration = 500;
        this.from = 0.0;
        this.to = 1.0;

        if (oncomplete != null) {
            this.oncomplete = oncomplete;
        }
    },

    animationComplete: function ()
    {
        delete this.animation;
        delete this.animator;

        this.post();
    
        if (this.oncomplete != null) {
            this.oncomplete();
        }

    //    iWLog("" + this.framecount / this.duration * 1000 + " fps");
    },

    stop: function ()
    {
        if (this.animation != null) {
            this.animator.stop();
            this.post();
            delete this.animation;
            delete this.animator;
        }
    },

    start: function ()
    {
        this.stop();

        this.pre();

        var self = this;

        var animator = new AppleAnimator(this.duration, 13);
        animator.oncomplete = function () { self.animationComplete(); };
        this.animator = animator;

        this.framecount = 0;

        var update = function (animation, now, first, done) {
            self.update(now);
            ++self.framecount;
        }
        this.animation = new AppleAnimation(this.from, this.to, update);
        animator.addAnimation(this.animation);
        animator.start();
    },

    pre: function ()
    {
        // Override for pre-animation setup
    },

    post: function ()
    {
        // Override for post-animation cleanup
    },

    update: function (now)
    {
        // Action goes here
    }
});

/* ------------------- */

var TransitionEffect = Class.create({
    initialize: function(current, next, oncomplete, shouldTighten)
    {
        this.current = current;
        this.next = next;
        this.oncomplete = oncomplete;
        this.shouldTighten = shouldTighten;
    
        this.effects = [
            { name: "Random", method: "random" }, // 0
            { name: "Jump", method: "jump" }, // 1
            { name: "Fade", method: "fade" }, // 2
            { name: "Wipe", directions: [ // 3
                { name: "Right", method: "wipeRight" }, // 3, 0
                { name: "Left", method: "wipeLeft" }, // 3, 1
                { name: "Down", method: "wipeDown" }, // 3, 2
                { name: "Up", method: "wipeUp" }, // 3, 3
                { name: "In", method: "wipeIn"}, // 3, 4
                { name: "Out", method: "wipeOut" } ] }, // 3, 5
            { name: "Close", directions: [ // 4
                { name: "Horizontal", method: "wipeCloseHoriz" }, // 4, 0
                { name: "Vertical", method: "wipeCloseVert" } ] }, // 4, 1
            { name: "Open", directions: [ // 5
                { name: "Horizontal", method: "wipeOpenHoriz" }, // 5, 0
                { name: "Vertical", method: "wipeOpenVert" } ] }, // 5, 1
            { name: "Reveal", directions: [ // 6
                { name: "Right", method: "slideOffRight" }, // 6, 0
                { name: "Left", method: "slideOffLeft" }, // 6, 1
                { name: "Down", method: "slideOffDown" }, // 6, 2
                { name: "Up", method: "slideOffUp" } ] }, // 6, 3
            { name: "Slide On", directions: [ // 7
                { name: "Right", method: "slideOnRight" }, // 7, 0
                { name: "Left", method: "slideOnLeft" }, // 7, 1
                { name: "Down", method: "slideOnDown" }, // 7, 2
                { name: "Up", method: "slideOnUp" } ] }, // 7, 3
            { name: "Push", directions: [ // 8
                { name: "Right", method: "pushRight" }, // 8, 0
                { name: "Left", method: "pushLeft" }, // 8, 1
                { name: "Down", method: "pushDown" }, // 8, 2
                { name: "Up", method: "pushUp" } ] }, // 8, 3
            { name: "Fade Through Black", method: "fadeThroughBlack" } // 9
        ];
    },

    doEffect: function (effect)
    {
        this[effect]();
    },

    random: function ()
    {
        // Dummied down version of random.  Since not all of the transitions are accessible through the UI, 
        // restrict random to the selected ones.  Also, since we always use direction 0, make random use that
        // direction as well.

        var supportedEffects = [
            2, // Fade
            6, // Reveal
            8, // Push
            9  // Fade Through Black
        ];
    
        var idx = Math.floor(Math.random() * supportedEffects.length);
        var num = supportedEffects[idx];
        var effect = this.effects[num];
        var method = effect.directions ? effect.directions[0].method : effect.method;

        this.doEffect(method);
    },

    animationComplete: function ()
    {
        delete this.animation;

        if (this.oncomplete != null) {
            this.oncomplete();
        }
    },

    jump: function ()
    {
        this.stop();
    
        this.next.upperShown();
        this.current.lowerHidden();
        this.animationComplete();
    },

    tighten: function (horiz, vert)
    {
        if(this.shouldTighten && (vert || horiz))
        {
            var top = 99999;
            var left = 99999;
            var right = 0;
            var bottom = 0;

            function expand(div)
            {
                for(var node = div.firstChild; node; node = node.nextSibling)
                {
                    var t = node.offsetTop;
                    var l = node.offsetLeft;

                    left = Math.min(left, l);
                    top = Math.min(top, t);
                    right = Math.max(right, l + node.offsetWidth);
                    bottom = Math.max(bottom, t + node.offsetHeight);
                }
            }

            expand(this.current.div);
            expand(this.next.div);

            if(vert == false || horiz == false)
            {
                var boxSize = this.current.boxSize();
                if(vert == false)
                {
                    top = 0;
                    bottom = boxSize.height;
                }
                if(horiz == false)
                {
                    left = 0;
                    right = boxSize.width;
                }
            }

            var frame = new IWRect(left, top, right - left, bottom - top);

            this.current.originalFrame = new IWRect(this.current.div.offsetLeft, this.current.div.offsetTop,
                                                    this.current.div.offsetWidth, this.current.div.offsetHeight);
            this.next.originalFrame = new IWRect(this.next.div.offsetLeft, this.next.div.offsetTop,
                                                this.next.div.offsetWidth, this.next.div.offsetHeight);

            this.current.offset = setFrameOptionallyMovingContents(this.current.div, frame, true);
            this.next.offset = setFrameOptionallyMovingContents(this.next.div, frame, true);
        }
    },

    loosen: function (loosenHoriz, loosenVert)
    {
        if(this.shouldTighten && (loosenHoriz || loosenVert))
        {
            // Reset the frames
            setFrameOptionallyMovingContents(this.current.div, this.current.originalFrame, false);
            setFrameOptionallyMovingContents(this.next.div, this.next.originalFrame, false);

            // And undo the offsetting done to their children
            offsetChildren(this.current.div, this.current.offset, true);
            offsetChildren(this.next.div, this.next.offset, true);
        }
    },

    fade: function ()
    {
        this.stop();

        var self = this;

        this.animation = new SimpleAnimation(function () { self.animationComplete(); });

        this.animation.pre = function ()
        {
            self.tighten(true, true);
            self.current.upperShown();
            self.next.lowerShown();
            self.current.setOpacity(1.0);
            self.next.setOpacity(1.0);
        }
        this.animation.post = function ()
        {
            self.loosen(true, true);
            self.next.upperShown();
            self.next.setOpacity(1.0);
            self.current.lowerHidden();
            self.current.setOpacity(1.0);
        }
        this.animation.update = function (now)
        {
            self.current.setOpacity(1.0 - now);
        }

        this.animation.start();
    },

    fadeThroughBlack: function ()
    {
        this.stop();

        var self = this;

        this.animation = new SimpleAnimation(function () { self.animationComplete(); });

        this.animation.pre = function ()
        {
            self.tighten(true, true);
            self.current.upperShown();
            self.next.lowerShown();
            self.current.setOpacity(1.0);
            self.next.setOpacity(0.0);
        }
        this.animation.post = function ()
        {
            self.loosen(true, true);
            self.next.upperShown();
            self.next.setOpacity(1.0);
            self.current.lowerHidden();
            self.current.setOpacity(1.0);
        }
        this.animation.update = function (now)
        {
            if(now < 0.5)
            {
                self.current.setOpacity((0.5 - now) * 2);
            }
            else
            {
                self.current.lowerHidden();
                self.next.setOpacity((now - 0.5) * 2);
            }
        }

        this.animation.start();
    },

    doWipe: function (wiper, inFlag, tightenHoriz, tightenVert)
    {
        this.stop();

        var self = this;

        this.animation = new SimpleAnimation(function () { self.animationComplete(); });

        this.animation.pre = function ()
        {
            self.current.resetSizeAndPosition();
            self.width = parseInt(self.current.div.style.width);
            self.height = parseInt(self.current.div.style.height);

            self.tighten(tightenHoriz, tightenVert);

            if (inFlag) {
                wiper.call(self, self.animation.from);
                self.next.upperShown();
                self.current.lowerShown();
            }
            else {
                self.current.upperShown();
                self.next.lowerShown();
            }
        }
        this.animation.post = function ()
        {
            self.loosen(tightenHoriz, tightenVert);
            self.next.upperShown();
            self.next.resetSizeAndPosition();
            self.current.lowerHidden();
            self.current.resetSizeAndPosition();
        }
        this.animation.update = function (now)
        {
            wiper.call(self, now);
        }

        this.animation.start();
    },

    wipeRight: function ()
    {
        this.doWipe(function (now)
        {
            this.current.setClipPx(0, this.width, this.height, now * this.width);
        });
    },

    wipeLeft: function ()
    {
        this.doWipe(function (now)
        {
            this.current.setClipPx(0, this.width - now * this.width, this.height, 0);
        });
    },

    wipeDown: function ()
    {
        this.doWipe(function (now)
        {
            this.current.setClipPx(now * this.height, this.width, this.height, 0);
        });
    },

    wipeUp: function ()
    {
        this.doWipe(function (now)
        {
            this.current.setClipPx(0, this.width, this.height - now * this.height, 0);
        });
    },

    wipeIn: function ()
    {
        this.doWipe(function (now)
        {
            var x = this.width * (1 - now) / 2;
            var y = this.height * (1 - now) / 2;
            this.next.setClipPx(y, this.width - x, this.height - y, x);
        }, true);
    },

    wipeOut: function ()
    {
        this.doWipe(function (now)
        {
            var x = this.width * now / 2;
            var y = this.height * now / 2;
            this.current.setClipPx(y, this.width - x, this.height - y, x);
        });
    },

    wipeCloseHoriz: function ()
    {
        this.doWipe(function (now)
        {
            var x = this.width * now / 2;
            this.current.setClipPx(0, this.width - x, this.height, x);
        });
    },

    wipeCloseVert: function ()
    {
        this.doWipe(function (now)
        {
            var y = this.height * now / 2;
            this.current.setClipPx(y, this.width, this.height - y, 0);
        });
    },

    wipeOpenHoriz: function ()
    {
        this.doWipe(function (now)
        {
            var x = this.width * (1 - now) / 2;
            this.next.setClipPx(0, this.width - x, this.height, x);
        }, true);
    },

    wipeOpenVert: function ()
    {
        this.doWipe(function (now)
        {
            var y = this.height * (1 - now) / 2;
            this.next.setClipPx(y, this.width, this.height - y, 0);
        }, true);
    },

    slideOffRight: function ()                                // <-- Reveal
    {
        this.doWipe(function (now)
        {
            var x = now * this.width;
            this.current.setLeftPx(x+(this.width-parseInt(this.current.div.style.width))/2);
        }, false, true, true);
    },

    slideOffLeft: function ()
    {
        this.doWipe(function (now)
        {
            var x = now * this.width;
            this.current.setClipPx(0, this.width, this.height, x);
            this.current.setLeftPx(-x);
        });
    },

    slideOffDown: function ()
    {
        this.doWipe(function (now)
        {
            var y = now * this.height;
            this.current.setClipPx(0, this.width, this.height - y, 0);
            this.current.setTopPx(y);
        });
    },

    slideOffUp: function ()
    {
        this.doWipe(function (now)
        {
            var y = now * this.width;
            this.current.setClipPx(y, this.width, this.height, 0);
            this.current.setTopPx(-y);
        });
    },

    slideOnRight: function ()
    {
        this.doWipe(function (now)
        {
            var x = (1 - now) * this.width;
            this.next.setClipPx(0, this.width, this.height, x);
            this.next.setLeftPx(-x);
        }, true);
    },

    slideOnLeft: function ()
    {
        this.doWipe(function (now)
        {
            var x = now * this.width;
            this.next.setClipPx(0, x, this.height, 0);
            this.next.setLeftPx(this.width - x);
        }, true);
    },

    slideOnDown: function ()
    {
        this.doWipe(function (now)
        {
            var y = (1 - now) * this.height;
            this.next.setClipPx(y, this.width, this.height, 0);
            this.next.setTopPx(-y);
        }, true);
    },

    slideOnUp: function ()
    {
        this.doWipe(function (now)
        {
            var y = now * this.height;
            this.next.setClipPx(0, this.width, y, 0);
            this.next.setTopPx(this.height - y);
        }, true);
    },

    pushRight: function ()                                      // <-- Push
    {
        this.doWipe(function (now)
        {
            var x = now * this.width;
            this.current.setLeftPx(x+(this.width-parseInt(this.current.div.style.width))/2);
            this.next.setLeftPx(x - this.width +(this.width-parseInt(this.current.div.style.width))/2);
        }, false, true, true);
    },

    pushLeft: function ()
    {
        this.doWipe(function (now)
        {
            var x = now * this.width;
            this.current.setClipPx(0, this.width, this.height, x);
            this.current.setLeftPx(-x);
            this.next.setClipPx(0, x, this.height, 0);
            this.next.setLeftPx(this.width - x);
        });
    },

    pushDown: function ()
    {
        this.doWipe(function (now)
        {
            var y = now * this.height;
            this.current.setClipPx(0, this.width, this.height - y, 0);
            this.current.setTopPx(y);
            this.next.setClipPx(this.height - y, this.width, this.height, 0);
            this.next.setTopPx(y - this.height);
        });
    },

    pushUp: function ()
    {
        this.doWipe(function (now)
        {
            var y = now * this.height;
            this.current.setClipPx(y, this.width, this.height, 0);
            this.current.setTopPx(-y);
            this.next.setClipPx(0, this.width, y, 0);
            this.next.setTopPx(this.height - y);
        });
    },

    stop: function ()
    {
        if (this.animation != null) {
            this.animation.stop();
            delete this.animation;
        }
    }
});

/* ------------------- */

var Slideshow = Class.create({
    initialize: function(container, photos, onchange, options)
    {
        // background (bogus?)
        this.reflectionHeight = 0;
        this.reflectionOffset = 0;
        this.backgroundColor = 'black';
        this.scaleMode = "fit";
        this.movieMode = kShowMovie;
        this.advanceAnchor = null;
        this.captionHeight = 0;
        this.shouldTighten = true;

        if(options)
        {
            if(options.reflectionHeight)
            {
                this.reflectionHeight = parseFloat(options.reflectionHeight);
            }
            if(options.reflectionOffset)
            {
                this.reflectionOffset = parseFloat(options.reflectionOffset);
            }
            if(options.backgroundColor)
            {
                this.backgroundColor = options.backgroundColor;
            }
            if(options.scaleMode)
            {
                this.scaleMode = options.scaleMode;
            }
            if(options.movieMode !== undefined)
            {
                this.movieMode = options.movieMode;
            }
            if(options.advanceAnchor !== undefined)
            {
                this.advanceAnchor = options.advanceAnchor;
            }
            if(options.captionHeight !== undefined)
            {
                this.captionHeight = options.captionHeight;
            }
            if(options.shouldTighten !== undefined)
            {
                this.shouldTighten = options.shouldTighten;
            }
        }

        this.background = new PhotoView(container, this.scaleMode, 0, 0, this.backgroundColor, false, false, 0);

        this.current = { view: new PhotoView(container, this.scaleMode, this.reflectionHeight, this.reflectionOffset, this.backgroundColor, this.movieMode, this.captionHeight) };
        this.prev = { view: new PhotoView(container, this.scaleMode, this.reflectionHeight, this.reflectionOffset, this.backgroundColor, this.movieMode, this.captionHeight) };
        this.next = { view: new PhotoView(container, this.scaleMode, this.reflectionHeight, this.reflectionOffset, this.backgroundColor, this.movieMode, this.captionHeight) };
        this.current.view.upperShown();
        this.prev.view.lowerHidden();
        this.next.view.lowerHidden();

        this.photos = photos;
        this.onchange = onchange;

        this.paused = true;
        this.photoDuration = 5000;
        this.currentPhotoNumber = 0;
        this.selectedTransition = "random";
        this.lastPhotoChange = new Date();
    },

    updateSize: function()
    {
        this.background.updateSize();
        this.current.view.updateSize();
        this.prev.view.updateSize();
        this.next.view.updateSize();
    
        this.p_updateAnchorSize();
    },

    transitionComplete: function (number)
    {
        this.transition.current.didAnimate();
        this.transition.next.didAnimate();

        this.transition.next.transitionComplete();

        delete this.transition;

        if (this.onchange != null) {
            this.onchange(number);
        }
    },

    showPhotoNumber: function (number, skipTransition)
    {
        this.p_showPhotoNumber(number, skipTransition, true);
    },

    p_updateAnchorSize: function()
    {
        if(this.advanceAnchor)
        {
            var box_size = this.current.view.boxSize();

            $(this.advanceAnchor).setStyle({
                position: "absolute",
                width: px(box_size.width),
                height: px(box_size.height)
            });
        
            if(windowsInternetExplorer)
            {
                // IE7 does not like the anchor to be empty.  As a workaround,
                // insert something into it.

                var img = this.advanceAnchor.firstChild;
                if(!img)
                {
                    img = document.createElement('img');
                    this.advanceAnchor.appendChild(img);
                }
                img.src = transparentGifURL();
                img.style.width = this.advanceAnchor.style.width;
                img.style.height = this.advanceAnchor.style.height;
            }
        }
    },

    p_setAdvanceAnchorHandler: function()
    {
        if(this.advanceAnchor)
        {        
            if(this.current.photo && this.current.photo.movieURL)
            {
                this.advanceAnchor.style.display = 'none';
                this.advanceAnchor.href = '';
                this.advanceAnchor.onclick = function() {return false;};
            }
            else
            {
                this.advanceAnchor.style.display = '';
                this.advanceAnchor.href = '#'+this.nextPhotoNumber();
                this.advanceAnchor.onclick = function(i) {
                    setTimeout(this.showPhotoNumber.bind(this, i, true), 0);
                }.bind(this, this.nextPhotoNumber());
            }
        }
    },

    p_showPhotoNumber: function (number, skipTransition, override)
    {
        this.cancelNextPhotoTimer();

        if(this.transition != null)
        {
            this.transition.stop();
            delete this.transition;
        }

        var transitionTime = 500;

        var current = this.current;
        var prev = this.prev;
        var next = this.next;

        // Get next.photo all set up
        if(next.photo == undefined || next.photo !== this.photos[number])
        {
            if(prev.photo != undefined && prev.photo === this.photos[number])
            {
                // iWLog("image is previous image, continuing");
                next = prev;
            }
            else if(override)
            {
                next.photo = this.photos[number];
            }
            else
            {
                return;
            }
        }

        // Make sure that next.photo.image is loaded, and load it if not
        if(next.photo.image.loaded() == false)
        {
            // iWLog("suspending, image not yet loaded");
            next.photo.image.load(arguments.callee.bind(this, number, skipTransition, false));
            return;
        }

        // Now that next.photo.image is loaded, make sure to stuff it into next.view
        if (next.view.image == undefined || next.view.image !== next.photo.image)
        {
            // iWLog("next.view.image is wrong, correcting");
            next.view.setImage(next.photo);
        }
        // iWLog("all clear, continuing");

        var self = this;

        current.view.willAnimate();
        next.view.willAnimate();

        this.transition = new TransitionEffect(current.view, next.view, function () { self.transitionComplete(number); }, this.shouldTighten);
        // If photos are advancing very quickly, use a jump transition
        if (skipTransition || (new Date()).getTime() - this.lastPhotoChange.getTime() < transitionTime)
        {
            this.transition.jump();
        }
        else
        {
            this.transition.doEffect(this.selectedTransition);
        }

        this.currentPhotoNumber = number;
        this.lastPhotoChange = new Date();

        this.current = next;

        // Set up the advance anchor
        this.p_setAdvanceAnchorHandler();
    
        // Preloading
        if (next === prev)
        {
            this.prev = this.next;
            this.next = current;
            // Went backwards - preload previous
            this.prev.photo = this.photos[this.prevPhotoNumber()];
            this.prev.photo.image.load(this.prev.view.setImage.bind(this.prev.view, this.prev.photo));
        }
        else
        {
            this.prev = current;
            this.next = prev;
            this.next.photo = this.photos[this.nextPhotoNumber()];
            this.next.photo.image.load(this.next.view.setImage.bind(this.next.view, this.next.photo));
        }

        this.startNextPhotoTimer();
    },

    start: function ()
    {
        this.paused = false;
        this.showPhotoNumber(0);
    },

    pause: function ()
    {
        if (!this.paused) {
            this.paused = true;
            this.cancelNextPhotoTimer();
        }
    },

    resume: function ()
    {
        if (this.paused) {
            this.paused = false;
            this.startNextPhotoTimer();
        }
    },

    inactivate: function()
    {
        // Inactive the slideshow.  Can happen, for example, when the detail view is exited.
        // Ensure that the slideshow is paused, and stop movie from playing (if there is a movie playing).
        this.pause();
        this.current.view.p_stopMovie();
    },

    startNextPhotoTimer: function (time)
    {
        if (this.paused) {
            return;
        }

        if (this.nextPhotoTimer != null) {
            (function(){return false}).assert("trying to set overlapping timer");
            return;
        }

        var self = this;

        var timedAdvance = function ()
        {
            delete self.nextPhotoTimer;

            var canAdvance = true;
            if (this.current.view.movieURL != null)
            {
                var movie = document[this.current.view.movieID];
                  
                var status = null;
                try
                {
                    status = movie.GetPluginStatus();
                }
                catch(e)
                {
                    status = null;
                }
            
                if(status == null || status.startsWith("Error:"))
                {
                    // An error occured. There's nothing we can do simply advance to the next entry                
                }
                else
                {
                    var time = movie.GetTime();
                    var duration = movie.GetDuration();
                    var timeScale = movie.GetTimeScale();
                    var timeRemaining = duration - time;
                    var timeRemainingInMS = timeRemaining / timeScale * 1000;
    
                    if(status == "Waiting" || status == "Loading")
                    {
                        // Wait until the movie is finished
                        canAdvance = false;
                        this.startNextPhotoTimer(timeRemainingInMS);
                    }
                    else if(status == "Playable" || status == "Complete")
                    {
                        if(timeRemainingInMS > 0)
                        {
                            // Still playing. Wait until it's finished.
                            canAdvance = false;
                            this.startNextPhotoTimer(timeRemainingInMS);
                        }
                    }
                }
            }
        
            if (canAdvance)
            {
                self.advance();
            }
        }.bind(this)

        this.nextPhotoTimer = setTimeout(timedAdvance, time || this.photoDuration);
    },

    cancelNextPhotoTimer: function ()
    {
        if (this.nextPhotoTimer != null) {
            clearTimeout(this.nextPhotoTimer);
            delete this.nextPhotoTimer;
        }
    },

    halt: function()
    {
        this.cancelNextPhotoTimer();
    },
    
    unhalt: function()
    {
        this.startNextPhotoTimer();
    },

    nextPhotoNumber: function ()
    {
        var number = this.currentPhotoNumber + 1;
        if (number >= this.photos.length) {
            number = 0;
        }

        return number;
    },

    prevPhotoNumber: function ()
    {
        var number = this.currentPhotoNumber - 1;
        if (number < 0) {
            number = this.photos.length - 1;
        }

        return number;
    },

    advance: function ()
    {
        this.showPhotoNumber(this.nextPhotoNumber());
    },

    goBack: function ()
    {
        this.showPhotoNumber(this.prevPhotoNumber());
    },

    setMovieTime: function (time)
    {
        this.current.view.setMovieTime(time);
    },

    setMovieParams: function (params)
    {
        this.current.view.setMovieParams(params);
    },

    setImage: function (image)
    {
        var current = this.current;

        current.photo.image = image;
        current.photo.image.load(current.view.setImage.bind(current.view, current.photo));
    },

    getAvailableTransitions: function ()
    {
        var te = new TransitionEffect;
        return te.effects;
    },

    setTransitionIndex: function (effectId, directionId)
    {
        var effects = this.getAvailableTransitions();
        var effect = effects[effectId];
        if (effect != null) {
            if (effect.directions == null) {
                if (effect.method != null) {
                    this.selectedTransition = effect.method;
                }
            }
            else {
                var dir = effect.directions[directionId];
                if (dir != null && dir.method != null) {
                    this.selectedTransition = dir.method;
                }
            }
        }
    }
});
//>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>


//<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<

var IWHorizontalAlignment =
{

kImageAlignLeft: 0,
kImageAlignCenter: 1,
kImageAlignRight: 2

}

var IWVerticalAlignment =
{

kImageAlignTop: 0,
kImageAlignMiddle: 1,
kImageAlignBottom: 2

}


var IWPhotoGridLayoutConstants =
{

kBorderThickness: 6.0,

vTextOffsetFromImage: 5.0,
vTextOffsetFromBottom: 10.0,

vTopSpacing: 5.0,

hMinSpacing: 15.0

}


var latestImageStream = null;
var latestIndex = null;

function IWStartSlideshow(url, imageStream, index, fullScreen)
{
    var windowWidth = 800;
    var windowHeight = 800;
    
    if (fullScreen)
    {
        // Fill the screen with the entire window
        windowWidth = screen.availWidth;
        windowHeight = screen.availHeight;
    }
    else
    {
        // Pick a reasonable window size based on the screen size.
        if (screen.availHeight > 975)
        {   
            windowWidth = 1000;
            windowHeight = 950;
        }
        else if (screen.availHeight > 775)
        {
            windowWidth = 800;
            windowHeight = 750;
        }
        else
        {
            windowWidth = 711;
            windowHeight = 533;
        }
    }

    windowWidth = Math.min(screen.availWidth, windowWidth);
    windowHeight = Math.min(screen.availHeight, windowHeight);
   
    var windowLeft = Math.round((screen.availWidth - windowWidth) / 2);
    var windowTop = Math.round((screen.availHeight - windowHeight) / 2) - 25;

    windowTop = Math.max(windowTop, 0);

    var slideWindow = window.open(url, 'slideshow',
            'scrollbars=no,titlebar=no,location=no,status=no,toolbar=no,resizable=no,width=' + windowWidth +
            ',height=' + windowHeight + ',top=' + windowTop + ',left=' + windowLeft);

    // Bug fix for Firefox: it seems to ignore you if you try to put the window too close to the menubar,
    // so we poke it if it's wrong.
    if (slideWindow.screenY && slideWindow.screenY > windowTop)
    {
        slideWindow.screenY = windowTop;
    }

    latestImageStream = imageStream;
    latestIndex = index;

    slideWindow.focus();

    return false;
}


function IWUpdateVerticalAlignment(element)    // TODO (mm) -- need better name
{
    element = $(element);
    
    function setVerticalAlignmentAndClearClass(element, valign)
    {
        var table = $(document.createElement("table"));
        table.setStyle({width: "100%", height: "100%", borderSpacing: 0});

        var tr = document.createElement("tr");
        table.appendChild(tr);

        var td = document.createElement("td");
        td.setAttribute("valign", valign);
        tr.appendChild(td);

        element.className = element.className.replace(new RegExp('\\bvalign-' + valign + '\\b'), "");

        element.parentNode.replaceChild(table, element);
        td.appendChild(element);

        if(windowsInternetExplorer)
        {
        // incredibly bogus IE workaround: set the table's parent's HTML to its own value or the text disappears (!)
            var html = table.parentElement.innerHTML;
            table.parentElement.innerHTML = html;
        }
    }

    var middleAligns = element.select('.valign-middle');
    for(var i = 0; i < middleAligns.length; ++i)
    {
        setVerticalAlignmentAndClearClass(middleAligns[i], "middle");
    }

    var bottomAligns = element.select('.valign-bottom');
    for(var i = 0; i < bottomAligns.length; ++i)
    {
        setVerticalAlignmentAndClearClass(bottomAligns[i], "bottom");
    }
}

function IWShowDiv(div, show)
{
    div.style.display = show ? "inline" : "none";
}

function IWToggleDetailView(showDetail, index, divID, headerViewID, footerViewID, detailViewID, shadow)
{
    var grid = $(divID);
    var header = $(headerViewID);
    var footer = $(footerViewID);
    var detail = $(detailViewID);

    var showGrid = !showDetail;

    function showHide()
    {
        IWShowDiv(grid, showGrid);
        IWShowDiv(header, showGrid);
        IWShowDiv(footer, showGrid);
        IWShowDiv(detail, showDetail);
    }

    if(showDetail)
    {        
        var detailWidget = widgets[detailViewID];

        function showDetailView()
        {
            showHide();
            detailWidget.willShow(index);

            IWSetSpacerHeight(grid, detailWidget.height());

            // Always scroll to the top of the window when displaying the detail view
            window.scrollTo(0,0);
        }

        if(index !== undefined)
        {
            detailWidget.showPhotoNumber(index, showDetailView);
        }
        else
        {
            showDetailView();
        }
    }
    else
    {
        showHide();

        if(isSafari && !isEarlyWebKitVersion && shadow)
        {
            // Safari requires that we reshadow images or they'll be empty
            // NOTE: It's important to NOT do this on IE since re-shadowing will result in an
            // explosion of memory usage
            // <rdar://problem/5253630> Clicking "Back to Album" will cause Windows IE to hang
            // NOTE: It's also important to skip this on early webkit versions, since the
            // shadow code is too slow (<rdar://problem/5271375>)
            $(divID).select('.framedImage').each(function(framed)
            {
                // Add the shadow rendering to the shared job queue to avoid freezing up the browser.
                var renderShadowJob = shadow.applyToElement.bind(shadow, framed);
                IWJobQueue.sharedJobQueue.addJob(renderShadowJob);
            });
        }

        if(grid.style.height)
        {
            IWSetSpacerHeight(grid, parseFloat(grid.style.height));
        }
    }
}

function IWSetSpacerHeight(photogrid, height)
{
    // make our section's spacer tall enough to force our footer down below our bottom.
    var parent = $(photogrid.parentNode);
    var spacers = parent.select('.spacer');
    var spacer = spacers[spacers.length - 1];

    if(initialSpacerHeight == 0)
    {
        initialSpacerHeight = parseInt(spacer.style.height);
    }

    var spacerHeight = Math.max(parseInt(photogrid.style.top) + height - spacer.offsetTop,
                                initialSpacerHeight);

    spacer.style.height = spacer.style.lineHeight = px(spacerHeight);
}


function IWLayoutPhotoGrid(divID, layout, stroke, imageStream, range, shadow, valignClass, opacity, slideshowParameters, slideshowURL, headerViewID, detailViewID, footerViewID, layoutOptions)
{
    if(range.length() == 0)
    {
        // Early exit if there is nothing to do.
        return;
    }
    
    // Clear any pending layout jobs (could cause problems if the user is navigating through a paginated
    // grid very quickly).
    IWJobQueue.sharedJobQueue.clearJobs();
    
    function toggleDetailViewNotification(notification)
    {
        var userInfo = notification.userInfo();
        IWToggleDetailView(userInfo.showDetailView, userInfo.index, divID, headerViewID, footerViewID, detailViewID, shadow);
    }

    var showPhotoIndex = undefined;

    var hash = location.hash;
    if(hash.length > 1)
    {
        // Only navigate to the detail view if the hash is a single integer that's in
        // the range of the imageStream (it's OK if the index is outside of the current 
        // display range).
        if(hash.match(/^\#(\d+)$/))
        {
            var idx = RegExp.$1;
            if(idx < imageStream.length)
            {
                showPhotoIndex = parseInt(idx);
            }
        }
    }

    if(isiPhone)
    {
        slideshowURL = slideshowURL.stringByDeletingLastPathComponent().stringByAppendingPathComponent("phoneshow.html");
    }

    slideshowURL = slideshowURL.stringByAppendingAsQueryString(slideshowParameters);

    if(detailViewID != null)
    {
        var detailView = widgets[detailViewID];
        if(!detailView.preferenceForKey("mediaStreamObject"))
        {
            if(showPhotoIndex !== undefined)
            {
                detailView.setPreferenceForKey(showPhotoIndex, "startIndex");
            }
    
            var mediaStreamObject = {load: function(baseURL, callback) { callback(imageStream); }};
            detailView.setPreferenceForKey(mediaStreamObject, "mediaStreamObject");
            NotificationCenter.addObserver(null, toggleDetailViewNotification, DetailViewToggleNotification, divID);
    
            if(slideshowParameters)
            {
                detailView.setPlaySlideshowFunction(IWStartSlideshow.bind(null, slideshowURL, imageStream, 0, slideshowParameters["fullScreen"]));
            }
        }
    }

    var photogrid = $(divID);

    photogrid.innerHTML = "";

    for(var i = 0; i < range.length(); ++i)
    {
        var frame = $(document.createElement('div'));
        var translation = layout.translationForTileAtIndex(i);
        var size = layout.tileSize();
        frame.style.cssText = iWPosition(true, translation.x, translation.y, size.width, size.height);
        frame.hide();

        photogrid.appendChild(frame);
    }

    var textBoxSize = layout.mTextBoxSize;

    var renderThumb = function(i, streamIndex)
    {
        var imageStreamEntry = imageStream[streamIndex];
        var wrapper = photogrid.childNodes[i];

        var textPos = layout.translationForTextAtIndexWithOffset(i, null);

        var textMarkup = "";
        if(textBoxSize.height > 0)
        {
            var richTitle = imageStreamEntry.richTitle();
            if(richTitle == null)
            {
                richTitle = imageStreamEntry.title();
            }
            var valignClassName = '';
            if(valignClass)
            {
                valignClassName = ' ' + valignClass;
            }

            var metric = imageStreamEntry.metric();
            if(metric != null && metric.length > 0)
            {
                metric = '<span class="metric">' + metric + '</span>';
            }
            else
            {
                metric = "";
            }

            var showTitle = layoutOptions ? layoutOptions.showTitle : true;
            var showMetric = layoutOptions ? layoutOptions.showMetric : true;

            var richCaption = '<div class="caption' + valignClassName + '">'+
                                  (showTitle ? ('<span class="title">' + richTitle + '</span>') : '') +
                                  ((showTitle && showMetric) ? '<br />' : '') +
                                  (showMetric ? metric : '') +
                              '</div>';

            var opacityMarkup = '';
            if(opacity < 0.999)
            {
                opacityMarkup = windowsInternetExplorer ? " filter: progid:DXImageTransform.Microsoft.Alpha(opacity=" + opacity * 100 + "); " : " opacity: " + opacity + "; ";
            }
            var overflowMarkup = "overflow: hidden; ";
            textMarkup =    '<div style="' + iWPosition(true, textPos.x, textPos.y, textBoxSize.width, textBoxSize.height) + opacityMarkup + overflowMarkup + '">' +
                                richCaption +
                            '</div>';
        }
        var thumbnailSize = imageStreamEntry.thumbnailNaturalSize();
        var scale = layout.p_scaleForImageOfSize(thumbnailSize);
        var scaledImageSize = thumbnailSize.scale(scale, scale, true);
        // <rdar://problems/5270759>
        // The thumbs should be the correct size, but unfortunately we need to deal with slight differences in the sizes here.
        // We don't use the scaled size unless it is significantly different than the thumbs natural size.
        if(Math.abs(scaledImageSize.width-thumbnailSize.width) <= 2 &&
           Math.abs(scaledImageSize.height-thumbnailSize.height) <= 2) 
        {
            scaledImageSize = thumbnailSize;
        }

        var frameMarkup = stroke.markupForImageStreamEntry(imageStreamEntry, scaledImageSize);

        wrapper.insert(frameMarkup);
        wrapper.insert(textMarkup);

        if(textMarkup.length > 0)
        {
            var textWrapper = wrapper.lastChild;
            textWrapper.style.zIndex = 1;
        }

        // fix up any vertically-aligned elements in our wrapper
        IWUpdateVerticalAlignment(wrapper);

        var framed = $(wrapper).selectFirst('.framedImage');

        var framePos = layout.translationForImageOfSizeAtIndexWithOffset(new IWSize(parseFloat(framed.style.width || 0), parseFloat(framed.style.height || 0)), i, null);

        framed.setStyle({left: px(framePos.x), top: px(framePos.y), zIndex: 1});

        if(detailViewID)
        {
            // create an anchor to link to this image and go to the detail view
            var anchor = $(document.createElement("a"));

            // position the anchor where framed is
            anchor.setStyle({
                position: "absolute",
                left: framed.style.left,
                top: framed.style.top,
                width: framed.style.width,
                height: framed.style.height,
                zIndex: 1
            });
            
            anchor.href = '#' + streamIndex;
            anchor.onclick = IWToggleDetailView.bind(null, true, streamIndex, divID, headerViewID, footerViewID, detailViewID, shadow);
            
            if(windowsInternetExplorer && (effectiveBrowserVersion >= 7))
            {
                // <rdar://problems/5102906>
                // IE7 does not like the anchor to be empty.  As a workaround,
                // insert something into it.
                var img = document.createElement('img');
                img.src = transparentGifURL();
                img.style.width = framed.style.width;
                img.style.height = framed.style.height;
                anchor.appendChild(img);
            }
            
            // insert the anchor after framed
            framed.parentNode.insertBefore(anchor, framed.nextSibling);
        }


        IWSetDivOpacity(framed, opacity);

        wrapper.show();

        if(shadow)
        {
            if(windowsInternetExplorer)
            {
            // BOGUS!!!! -- handle the shadow here for IE.  We should be using
            //              IWShadow for this, but it doesn't really work (see
            //              rdar://problems/4691725&5102936)
                var imgs = framed.select('img');

                // find the extents of the images
                var left = 0, top = 0;
                var right = parseFloat(framed.style.width || 0);
                var bottom = parseFloat(framed.style.height || 0);
                for(var e = 0; e < imgs.length; ++e)
                {
                    var style = imgs[e].style;

                    var l = parseFloat(style.left || 0);
                    var t = parseFloat(style.top || 0);

                    left = Math.min(l, left);
                    top = Math.min(t, top);
                    right = Math.max(l + parseFloat(style.width || 0), right);
                    bottom = Math.max(t + parseFloat(style.height || 0), bottom);
                }

                // grow framed large enough to hold its images.  If we don't do this,
                //  framed's contents get clipped in IE6.  We also need to do something
                //  like this when we have a shadow and we render into a canvas.
                framed.setStyle({
                    left: px(parseFloat(framed.style.left || 0) + left),
                    top: px(parseFloat(framed.style.top || 0) + top),
                    width: px(right - left),
                    height: px(bottom - top)
                });

                // move framed's children so that they're where they used to be (we moved
                //  framed's left and top left and up, so move his children right and
                //  down)
                var framedChildren = framed.childNodes;
                for(var c = 0; c < framedChildren.length; ++c)
                {
                    var child = framedChildren[c];
                    if(child.nodeType == Node.ELEMENT_NODE)
                    {
                        child.style.left = px(parseFloat(child.style.left || 0) - left);
                        child.style.top = px(parseFloat(child.style.top || 0) - top);
                    }
                }

                var blurRadius = shadow.mBlurRadius * .75;

                var xOffset = shadow.mOffset.x - blurRadius;
                var yOffset = shadow.mOffset.y - blurRadius;

                var shadowDiv = framed.cloneNodeExcludingIDs(true);

                shadowDiv.style.left = px(framePos.x + xOffset);
                shadowDiv.style.top = px(framePos.y + yOffset);

                shadowDiv.style.filter = "progid:DXImageTransform.Microsoft.MaskFilter()" +
                                         " progid:DXImageTransform.Microsoft.MaskFilter(color=" + shadow.mColor + ")" +
                                         " progid:DXImageTransform.Microsoft.Blur(pixelradius=" + blurRadius + ")" +
                                         " progid:DXImageTransform.Microsoft.Alpha(opacity=" + shadow.mOpacity * opacity * 100 + ")";

                framed.parentNode.insertBefore(shadowDiv, framed);
            }
            else
            {
                // Early versions of webkit (before 3.0) are really, really slow when dealing with more than a
                // few shadows.  Thus, we disable them on photogrids.
                // <rdar://5250448> Goldenrod: Photos: Images take a long time to appear on published page
                if (!isEarlyWebKitVersion)
                {
                    shadow.applyToElement(framed);
                }
            }
        }

        imageStreamEntry.didInsertThumbnailMarkupIntoDocument();
        imageStreamEntry.unloadThumbnail();
    };

    for(var i = 0; i < range.length(); ++i)
    {
        var streamIndex = range.location() + i;
        var renderThumbJob = renderThumb.bind(null, i, streamIndex);
        imageStream[streamIndex].loadThumbnail(IWJobQueue.prototype.addJob.bind(IWJobQueue.sharedJobQueue, renderThumbJob));
    }

    var headerView = widgets[headerViewID];
    if(headerView)
    {
        if(slideshowParameters)
        {
            // set the header view's start slideshow button
            headerView.setPlaySlideshowFunction(IWStartSlideshow.bind(null, slideshowURL, imageStream, 0, slideshowParameters["fullScreen"]));
        }
        headerView.setPreferenceForKey(imageStream.length, "entryCount");
    }

    function setFooterOffset(height)
    {
        var footer = $(footerViewID);
        var footerOffsetTop = photogrid.offsetTop + height - layout.mBottomPadding;
        footer.style.top = px(footerOffsetTop);
    }

    var gridHeight = layout.totalHeightForCount(range.length());
    setFooterOffset(gridHeight);
    IWSetSpacerHeight(photogrid, gridHeight);
    photogrid.style.height = px(gridHeight);

    if(showPhotoIndex !== undefined)
    {
        IWToggleDetailView(true, showPhotoIndex, divID, headerViewID, footerViewID, detailViewID, shadow);
    }
}

var initialSpacerHeight = 0;

var IWPhotoGridLayout = Class.create({
    initialize: function(columnCount, maxImageSize, textBoxSize, tileSize, topPadding, bottomPadding, spacing, framePadding)
    {
        this.mColumnCount = columnCount;
        this.mMaxImageSize = maxImageSize;
        this.mTextBoxSize = textBoxSize;
        this.mTileSize = tileSize;
        this.mTopPadding = topPadding;
        this.mBottomPadding = bottomPadding;
        this.mSpacing = spacing;
        this.mFramePadding = framePadding;

        this.mCachedDataValid = true;
    },

    tileSize: function()
    {
        return this.mTileSize;
    },

    translationForTileAtIndex: function(index)
    {
        var tileSize = this.mTileSize;
        var columnCount = this.mColumnCount;
        var translation = new IWPoint(tileSize.width * (index % columnCount) + IWPhotoGridLayoutConstants.kBorderThickness,
                                      tileSize.height * Math.floor(index / columnCount) + IWPhotoGridLayoutConstants.kBorderThickness + this.mTopPadding + IWPhotoGridLayoutConstants.vTopSpacing);
        return translation;
    },

    translationForImageOfSizeAtIndexWithOffset: function(imageSize, index, offset)
    {
        // translate for tile position
        var tileSize = this.mTileSize;
        var translation = new IWPoint(0, 0);
        var imageAreaSize = new IWSize(tileSize.width,
                                       tileSize.height - (IWPhotoGridLayoutConstants.vTextOffsetFromImage + this.mTextBoxSize.height + IWPhotoGridLayoutConstants.vTextOffsetFromBottom));

        // offset translation for horizontal alignment
        translation.x += (imageAreaSize.width - imageSize.width) / 2;

        // offset translation for vertical alignment
        translation.y += (imageAreaSize.height - imageSize.height);

        // offset translation for (animation) offset
        if(offset != null)
        {
            translation.x += offset.translation.width;
            translation.y += offset.translation.height;
        }

    // TODO (mm) -- offset to factor in the frame width

        return translation;
    },

    translationForTextAtIndexWithOffset: function(index, offset)
    {
        var tileSize = this.mTileSize;
        var textBoxSize = this.mTextBoxSize;
        var translation = new IWPoint((tileSize.width - textBoxSize.width) / 2.0,
                                      (tileSize.height - textBoxSize.height) - IWPhotoGridLayoutConstants.vTextOffsetFromBottom);

        if(offset != null)
        {
            translation.x += offset.translation.width;
            translation.y += offset.translation.height;
        }

        return translation;
    },

    totalHeightForCount: function(count)
    {
        // TODO (mm) -- do something a little more classy here
        if(count == 0)
            count = 1;

        var columnCount = this.mColumnCount;
        var lastRow = Math.floor((count + columnCount - 1) / columnCount);
        var tileSize = this.mTileSize;

        return tileSize.height * lastRow + IWPhotoGridLayoutConstants.kBorderThickness * 2 + this.mTopPadding + this.mBottomPadding;
    },

    p_scaleForImageOfSize: function(imageSize)
    {
        var maxImageSize = this.mMaxImageSize;

        return Math.min((maxImageSize.width - this.mFramePadding.width) / imageSize.width,
                        (maxImageSize.height - this.mFramePadding.height) / imageSize.height);
                            
    }
});

///////////////////////////////////////////////////////////////
// IWJobQueue
//
// General purpose job queue.  Each 'job' consists of a method
// to be called (all parameters to the method must be fully
// bound).  The job queue will run jobs in the order in which they
// are queued up, and will run them in small batches in order
// to give time to the browser for layout, etc.
// Anything that needs to run "sometime" but that can take a long
// time to run, can be thrown into the job queue to ensure that
// the browser doesn't become unresponsive for an extended period of
// time.

var IWJobQueue = Class.create({
    initialize: function()
    {
        this.mJobQueue = [];
        this.mTimeout = null;
    },

    addJob: function(job)
    {
        this.mJobQueue.push(job);
    
        // Set a timeout to run the queued jobs, since there will now be at least one job to run.
        // This method can be called multiple times before any of the jobs are actually run,
        // so reset the timeout on each call rather than setting multiple timeouts.
        this.p_setTimeout();
    },

    clearJobs: function(job)
    {
        this.p_cancelTimeout();
        this.mJobQueue = [];
    },

    p_runQueuedJobs: function()
    {
        // In order to prevent the page load from coming to a complete halt while dynamic effects are applied,
        // we try to play nice when applying a large number of effects.  We apply as many effects as we can
        // until we exceed a duration of 100ms, then set a timeout to apply the remainder.

        this.p_cancelTimeout();
    
        var startTime = new Date().getTime();
        var duration = 0;
    
        while(this.mJobQueue.length > 0 && duration < 100)
        {
            // Remove one entry from the queue
            var job = this.mJobQueue.shift();

            // Run the job
            if(job)
            {
                job();
            }

            // Update the duration
            duration = (new Date().getTime()) - startTime;
        }
    
        if(this.mJobQueue.length > 0)
        {
            // Still have jobs in the queue -- set a timeout to run the rest of them
            this.p_setTimeout();
        }
    },

    p_cancelTimeout: function()
    {
        if(this.mTimeout != null)
        {
            clearTimeout(this.mTimeout);
            this.mTimeout = null;
        }
    },

    p_setTimeout: function()
    {
        if(this.mTimeout == null)
        {
            this.mTimeout = setTimeout(this.p_runQueuedJobs.bind(this), 0);
        }
    }
});

IWJobQueue.sharedJobQueue = new IWJobQueue();
//>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>


//<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
////////////////////////////////////////////////////////////////////////////////
//
//  iWeb - AppleAnimator.js
//  Copyright (c) 2005-2007 Apple Inc. All rights reserved.
//
////////////////////////////////////////////////////////////////////////////////

/*
IMPORTANT:  This Apple software and the associated images located in
/System/Library/WidgetResources/AppleClasses/ (collectively "Apple Software")
are supplied to you by Apple Computer, Inc. (“Apple”) in consideration of your
agreement to the following terms. Your use, installation and/or redistribution
of this Apple Software constitutes acceptance of these terms. If you do not
agree with these terms, please do not use, install, or redistribute this Apple
Software.

In consideration of your agreement to abide by the following terms, and subject
to these terms, Apple grants you a personal, non-exclusive license, under
Apple’s copyrights in the Apple Software, to use, reproduce, and redistribute
the Apple Software, in text form (for JavaScript files) or binary form (for
associated images), for the sole purpose of creating Dashboard widgets for Mac
OS X.

If you redistribute the Apple Software, you must retain this entire notice and
the warranty disclaimers and limitation of liability provisions (last two
paragraphs below) in all such redistributions of the Apple Software.

You may not use the name, trademarks, service marks or logos of Apple to endorse
or promote products that include the Apple Software without the prior written
permission of Apple. Except as expressly stated in this notice, no other rights
or licenses, express or implied, are granted by Apple herein, including but not
limited to any patent rights that may be infringed by your products that
incorporate the Apple Software or by other works in which the Apple Software may
be incorporated.

The Apple Software is provided on an "AS IS" basis.  APPLE MAKES NO WARRANTIES,
EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF
NON-INFRINGEMENT, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE,
REGARDING THE APPPLE SOFTWARE OR ITS USE AND OPERATION ALONE OR IN COMBINATION
WITH YOUR PRODUCTS.

IN NO EVENT SHALL APPLE BE LIABLE FOR ANY SPECIAL, INDIRECT, INCIDENTAL OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
ARISING IN ANY WAY OUT OF THE USE, REPRODUCTION, AND/OR DISTRIBUTION OF THE
APPLE SOFTWARE, HOWEVER CAUSED AND WHETHER UNDER THEORY OF CONTRACT, TORT
(INCLUDING NEGLIGENCE), STRICT LIABILITY OR OTHERWISE, EVEN IF APPLE HAS BEEN
ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

var AppleAnimator = Class.create({
    initialize: function(duration, interval, optionalFrom, optionalTo, optionalCallback)
    {
        this.startTime = 0;
        this.duration = duration;
        this.interval = interval;
        this.animations = new Array;
        this.timer = null;
        this.oncomplete = null;
    
        this._firstTime = true;
    
        var self = this;
    
        this.animate = function (self) {
        
            function limit_3 (a, b, c) {
                return a < b ? b : (a > c ? c : a);
            }
        
            var T, time;
            var ease;
            var time  = (new Date).getTime();
            var dur = self.duration;
            var done;
                
            T = limit_3(time - self.startTime, 0, dur);
            time = T / dur;
            ease = 0.5 - (0.5 * Math.cos(Math.PI * time));
        
            done = T >= dur;
        
            var array = self.animations;
            var c = array.length;
            var first = self._firstTime;
        
            for (var i = 0; i < c; ++i)
            {
                array[i].doFrame (self, ease, first, done, time);
            }
        
            if (done)
            {
                self.stop();
                if  (self.oncomplete != null)
                {
                    self.oncomplete();
                }
            }
        
            self._firstTime = false;
        }

        if (optionalFrom !== undefined && optionalTo !== undefined && optionalCallback !== undefined)
        {
            this.addAnimation(new AppleAnimation (optionalFrom, optionalTo, optionalCallback));
        }
    },

    start: function () {
        if (this.timer == null)
        {
            var timeNow = (new Date).getTime();
            var interval = this.interval;
        
            this.startTime = timeNow - interval; // see it back one frame
        
            this.timer = setInterval (this.animate.bind(null, this), interval);
        }
    },

    stop: function () {
        if (this.timer != null)
        {
            clearInterval(this.timer);
            this.timer = null;
        }
    },

    addAnimation: function (animation) {
        this.animations[this.animations.length] = animation;
    }
});

//
// Animation class
//

var AppleAnimation = Class.create({
    initialize: function(from, to, callback)
    {
        this.from = from;
        this.to = to;
        this.callback = callback;
    
        this.now = from;
        this.ease = 0;
        this.time = 0;
    },

    doFrame: function (animation, ease, first, done, time) {
        var now;
    
        if (done)
            now = this.to;
        else
            now = this.from + (this.to - this.from) * ease;
    
        this.now = now;
        this.ease = ease;
        this.time = time;
        this.callback (animation, now, first, done);
    }
});

//>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>


//<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<

// Fetches the comment summary info for the specified URL and calls the callback with a dictionary
// containing the results.
//
function IWCommentSummaryInfoForURL(resourceURL, callback)
{
    function handleSummaryData(request)
    {
        var result = {};
        
        if (request.responseText)
        {
            var r = request.responseText.match(/.*= ((true)|(false));.*\n.*= (\d+)/);
            if (r)
            {
                result.enabled = (r[1] == "true");
                result.count = Number(r[4]);
            }
        }
        
        // Perform callback.  On failure, the callback is still called, but not from here.
        callback(result);
    }

    var summaryURL = resourceURL + "?wsc=summary.js&ts=" + (new Date().getTime());
    new Ajax.Request(summaryURL, {
        method: 'get',
        onSuccess: handleSummaryData,
        onFailure: callback.bind(null, {})
    });
}

// Fetches the comment summary info and calls the callback with the number of comments.
//
function IWCommentCountForURL(resourceURL, callback)
{
    function myCallback(info)
    {
        if (info.count === undefined)
            info.count = 0;
        callback(info.count);
    }
    IWCommentSummaryInfoForURL(resourceURL, myCallback);
}

//>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
