//
//  iWeb - WidgetCommon.js
//  Copyright (c) 2007-2008 Apple Inc. All rights reserved.
//

// Global array of available widgets.  Widget() constructor adds a widget to this array.
var widgets = [];
var identifiersToStringLocalizations = [];

var Widget = Class.create({
    initialize: function(instanceID, widgetPath, sharedPath, sitePath, preferences, runningInApp)
    {
        // Arguments:
        //      instanceID      unique identifier for this instance of the widget in this page
        //                      (there should be a div with this ID in the document, which contains our content)
        //      widgetPath      same as $WIDGET_PATH -- a URL to the widget's directory
        //      sharedPath      same as $WIDGET_SHARED_PATH -- a URL to resources shared across widgets
        //      sitePath        a URL to the site that this widget is in
        //      preferences     object containing this widget's persistent state
        //      runningInApp    boolean, true if the widget is running inside iWeb
        //
        // If there is no instanceID, assume this object is in use as a prototype
        // of a subclass, and don't do anything.

        if (instanceID)
        {
            this.instanceID = instanceID;
            this.widgetPath = widgetPath;
            this.sharedPath = sharedPath;
            this.sitePath = sitePath;
            this.preferences = preferences;
            this.runningInApp = ( runningInApp === undefined ) ? false : runningInApp;
            this.onloadReceived = false;

            if (this.preferences && this.runningInApp == true)
            {
                this.preferences.widget = this; // for the benefit of running in iWeb, so it can call this.changedPreferenceForKey()
                setTransparentGifURL(this.sharedPath.stringByAppendingPathComponent("None.gif"));
            
            }
    
            // stash ourselves where other things can find us
            this.div().widget = this;       // go from div to widget
            window[instanceID] = this;      // give widget a global name
            widgets.push(this);             // put widget in widgets[n]
            widgets[instanceID] = this;     // put widget in widgets[instanceID]

            if (!this.constructor.instances)
            {
                this.constructor.instances = new Array();
            }
            this.constructor.instances.push(this);  // put widget in WidgetSubclass.instances[]
        }
    },

    div: function()
    {
        var divID = this.instanceID;
        if (arguments.length == 1)
        {
            divID = this.instanceID + "-" + arguments[0];
        }
        return $(divID);
    },

    onload: function() 
    {
        this.onloadReceived = true;
    },

    onunload: function() 
    {
    },

    didBecomeSelected: function()
    {
    },

    didBecomeDeselected: function()
    {
    },

    didBeginEditing: function()
    {
    },

    didEndEditing: function()
    {
    },

    setNeedsDisplay: function()
    {
        // iWeb will call on certain widgets when it wants them to redraw
    },

    preferenceForKey: function(key)
    {
      var value;
      if (this.preferences)
        value = this.preferences[key];
      return value;   
    },

    initializeDefaultPreferences: function(prefs)
    {
        var self = this;
        $H(prefs).each(function(pair)
        {
            if (self.preferenceForKey(pair.key) === undefined)
            {
                self.setPreferenceForKey(pair.value, pair.key, false);
            }
        });
    },

    setPreferenceForKey: function(preference, key, registerUndo/*=true*/)
    {
        if (this.runningInApp)
        {
            if (registerUndo === undefined) 
                registerUndo = true;
            
            if ((registerUndo == false) && this.preferences.disableUndoRegistration)
                this.preferences.disableUndoRegistration();

            this.preferences[key] = preference;
            // Host will send changePref notification, so JS doesn't need to.
        
            if ((registerUndo == false) && this.preferences.enableUndoRegistration)
                this.preferences.enableUndoRegistration();
        }
        else
        {
            this.preferences[key] = preference;
            this.changedPreferenceForKey(key);
        }
    },

    changedPreferenceForKey: function(key)
    {
    },

    // This is now iWeb posts notifications to widgets.
    postNotificationWithNameAndUserInfo: function(name, userInfo)
    {
        if (window.NotificationCenter !== undefined)
        {
            NotificationCenter.postNotification(new IWNotification(name, null, userInfo));
        }
    },

    sizeWillChange: function()
    {
    },

    sizeDidChange: function()
    {
    },

    widgetWidth: function()
    {
        var enclosingDiv = this.div();
        if (enclosingDiv)
            return enclosingDiv.offsetWidth;
        else
            return null;
    },

    widgetHeight: function()
    {
        var enclosingDiv = this.div();
        if (enclosingDiv)
            return enclosingDiv.offsetHeight;
        else
            return null;
    },

    // Creates a widget-instance-specific ID constructed from the instance ID and one or two parameters
    // as follows:
    //
    //  <$WIDGET_ID> + '-' + <param1>
    //  <$WIDGET_ID> + '-' + <param1> + '$' + <param2>
    //
    // The first form constructs ids for nodes scoped to the widget ID, while the second form
    // can be used to construct indexed versions of these ids, for use when a template node is
    // cloned and ids are adjusted with adjustNodeIds().
    //
    getInstanceId: function(id)
    {
        var fullId = this.instanceID + "-" + id;
        if (arguments.length == 2)
        {
            fullId += ("$" + arguments[1]);
        }
        return fullId;
    },

    // Perform document.getElementById with a node id constructed with getInstanceId as describe
    // above.  Returns an extended element.
    //
    getElementById: function(id)
    {
        var fullId = this.getInstanceId.apply(this, arguments);
        return $(fullId);
    },

    localizedString: function(string)
    {
        return LocalizedString(this.widgetIdentifier, string);
    },

    // This utility method depends on this widget having its 'm_views' ivar set
    // to a hash of View objects.  See the RSS feed widget.
    showView: function(viewName)
    {
        var futureView = this.m_views[viewName];
        if((futureView != this.m_currentView) && (futureView != this.m_futureView))
        {
            this.m_futureView = futureView;
            if(this.m_fadeAnimation)
            {
                this.m_fadeAnimation.stop();
            }

            var previousView = this.m_currentView;
            this.m_currentView = futureView;
            var currentView = this.m_currentView;
            this.m_futureView = null;

            this.m_fadeAnimation = new SimpleAnimation( function() { delete this.m_fadeAnimation;}.bind(this) );

            this.m_fadeAnimation.pre = function()
            {
                if(previousView)
                {
                    previousView.ensureDiv().setStyle({zIndex: 0, opacity: 1});
                }
                if(currentView)
                {
                    currentView.ensureDiv().setStyle({zIndex: 1, opacity: 0});
                    currentView.show();
                    currentView.render();
                }
            }

            this.m_fadeAnimation.post = function()
            {
                !previousView || previousView.hide();
                !currentView || currentView.ensureDiv().setStyle({zIndex: '', opacity: 1});
                !currentView || !currentView.doneFadingIn || currentView.doneFadingIn();
            }

            this.m_fadeAnimation.update = function(now)
            {
                !currentView || currentView.ensureDiv().setOpacity(now);
                !previousView || previousView.ensureDiv().setOpacity(1-now);
            }.bind(this);

            this.m_fadeAnimation.start();
        }
    }
});

// Widget.onload() and Widget.onunload() should be called from the body's
// onload and onunload handlers.

Widget.onload = function()
{
    for (var i=0; i < widgets.length; i++)
    {
        widgets[i].onload();
    }
}

Widget.onunload = function()
{
    for (var i=0; i < widgets.length; i++)
    {
        widgets[i].onunload();
    }
}


function RegisterWidgetStrings(identifier, strings)
{
    identifiersToStringLocalizations[identifier] = strings;
}

function LocalizedString(identifier, string)
{
    var localized = undefined;

    var localizations = identifiersToStringLocalizations[identifier];
    if(localizations === undefined)
    {
        iWLog("warning: no localizations for widget " + identifier + ", (key:" + string + ")");
    }
    else
    {
        localized = localizations[string];
    }

    if(localized === undefined)
    {
        iWLog("warning: couldn't find a localization for '" + string + "' for widget " + identifier);
        localized = string;
    }

    return localized;
}

function WriteLocalizedString(identifier, string)
{
    document.write(LocalizedString(identifier, string));
}

var JSONFeedRendererWidget = Class.create(Widget, {

    initialize: function($super, instanceID, widgetPath, sharedPath, sitePath, preferences, runningInApp)
    {
        if (instanceID)
        {
            $super(instanceID, widgetPath, sharedPath, sitePath, preferences, runningInApp);
        }
    },

    changedPreferenceForKey: function(key)
    {
        try
        {
            var value = this.preferenceForKey(key);
            if (key == "sfr-shadow")
            {
                if (value != null)
                {
                    this.sfrShadow = eval(value);
                }
                else
                {
                    this.sfrShadow = null;
                }
                // Render immediately because this is important for interactive shadow parameter
                // adjustments.
                this.renderFeedItems("sfr-shadow");
            }
    
            if (key == "sfr-stroke")
            {
                if (value !== null)
                    this.sfrStroke = eval(value);
                else
                    this.sfrStroke = null;
                this.invalidateFeedItems("sfr-stroke");
            }
    
            if (key == "sfr-reflection")
            {
                if (value !== null)
                {
                    this.sfrReflection = eval(value);
                }
                else
                {
                    this.sfrReflection = null;
                }
                this.invalidateFeedItems("sfr-reflection");
            }
        }
        catch (e)
        {
            iWLog("JSONFeedRendererWidget: exception");
            debugPrintException(e);
        }
    },
    
    invalidateFeedItems: function(reason)
    {
        trace('invalidateFeedItems(%s)', reason);
        if (this.pendingRender !== null)
        {
            clearTimeout(this.pendingRender);
        }
    
        this.pendingRender = setTimeout(function() 
                                        {
                                            this.pendingRender = null; 
                                            this.renderFeedItems(reason); 
                                        }.bind(this), 50);
    },

    rerenderImage: function(imgGroupDiv, imgDiv, imageUrlString, entryHasImage, photoProportions, imageWidth, positioningHandler, onloadHandler)
    {
        imgGroupDiv.update(); // remove all contents
        if (entryHasImage)
        {
            // These flags guard against multiple applications of effects which can happen as a result of
            // interactive manipulation of image parameters combined with deferred rendering due to layout
            // uncertainty. Specifically: dragging the excerpt size slider can cause many calls through
            // here which effectively queues a number of applyEffects calls (because the layout information
            // isn't available to apply the effect immediately).
            // Ref: <rdar://problem/5069541> Frames are doubled after using Excerpt Length slider
            //
            imgGroupDiv.strokeApplied     = false;
            imgGroupDiv.reflectionApplied = false;
            imgGroupDiv.shadowApplied     = false;

            // Reset the margins on the group div, because adding a stroke may change them.
            imgGroupDiv.setStyle({
                marginLeft: 0,
                marginTop: 0,
                marginRight: 0,
                marginBottom: 0
            });
    
            // IE 6 likes position to explicitly be set to relative
            imgGroupDiv.style.position   = 'relative'; 
            imgDiv.style.position = 'relative';

            var imageUrl = imageUrlString || transparentGifURL();

            var image = IWCreateImage(imageUrl);

            image.load(function(image, imgDiv, imgGroupDiv, positioningHandler, onloadHandler)
            {
                var cropDiv = this.croppingDivForImage(image, photoProportions, imageWidth);
                imgGroupDiv.appendChild(cropDiv);
                
                if (positioningHandler)
                {
                    positioningHandler();
                }
        
                // Don't apply effects to the transparent image
                if (image.sourceURL() !== transparentGifURL())
                {
                    this.applyEffects(imgGroupDiv);
                }

                if (onloadHandler) 
                {
                    onloadHandler();
                }
            }.bind(this, image, imgDiv, imgGroupDiv, positioningHandler, onloadHandler));
        }
    },

    croppingDivForImage: function(image, kind, width)
    {
        var croppedSize = function(originalSize, cropKind, width)
        {
            if (cropKind == "Square")
            {
                return new IWSize(width, width);
            }
            else if (cropKind == "Landscape")
            {
                return new IWSize(width, width * (3/4));
            }
            else if (cropKind == "Portrait")
            {
                return new IWSize(width, width * (4/3));
            }
            else
            {
                var scaleFactor = width / originalSize.width;
                return originalSize.scale(scaleFactor, scaleFactor, true);
            }
        };

        var cropDiv = null;

        if (image.loaded())
        {
            var img = $(document.createElement('img'));
            img.src = image.sourceURL();
            var natural = image.naturalSize();
        
            cropDiv = $(document.createElement("div"));
            cropDiv.appendChild(img);

            var croppingDivForImage_helper = function ( loadedImage ) 
            {
                if (loadedImage)
                {
                    natural = new IWSize(loadedImage.width, loadedImage.height); 
                }
            
                var cropped = croppedSize(natural, kind, width);
                var scaleFactor = cropped.width / natural.width;
                if (natural.aspectRatio() > cropped.aspectRatio())
                {
                    scaleFactor = cropped.height / natural.height;
                }
                var scaled = natural.scale(scaleFactor);
                var offset = new IWPoint(Math.abs(scaled.width - cropped.width) / 2,
                                         Math.abs(scaled.height - cropped.height) / 2);
            
                img.setStyle({
                    width: px(scaled.width),
                    height: px(scaled.height),
                    marginLeft: px(-offset.x),
                    marginTop: px(-offset.y),
                    position: 'relative'
                });
            
                cropDiv.setStyle({
                    width: px(cropped.width),
                    height: px(cropped.height),
                    overflow: "hidden",
                    position: 'relative'
                });
                    
                cropDiv.className = "crop";
            }
        
            if(windowsInternetExplorer && effectiveBrowserVersion < 7 && img.src.indexOf(transparentGifURL()) != -1)
            {            
                var originalImage = new Image();
                originalImage.src = img.originalSrc;
                if (originalImage.complete)
                {
                    croppingDivForImage_helper(originalImage); 
                }
                else
                {
                    originalImage.onload = croppingDivForImage_helper.bind(null, originalImage); 
                }
            }
            else
            {
                croppingDivForImage_helper( null );
            }
        }
        return cropDiv;
    },

    // Apply stroke, reflection and shadow effects to the specified div. These methods require the offsetWidth
    // and offsetHeight for the target DIV to have valid values, but sometimes after DOM manipulation this can 
    // take a while to happen. So, if we determine that these values are undefined, then defer the action of
    // this function for a short time and try again later. The short delay gives the layout engine time to
    // do whatever it's doing and make the values available.
    //
    // Note: This means that it is important for subsequent code to understand that calling this method may
    // not do anything immediately.
    //
    applyEffects: function(div)
    {
        if (this.sfrShadow || this.sfrReflection || this.sfrStroke)
        {
            // Safari: offsetWidth/height undefined until layout occurs
            // IE: offsetWidth/height zero until layout occurs when revisiting a cached page
            if ((div.offsetWidth === undefined) || (div.offsetHeight === undefined) ||
                (div.offsetWidth === 0) || (div.offsetHeight === 0))
            {
                setTimeout(JSONFeedRendererWidget.prototype.applyEffects.bind(this, div), 0)
                return;
            }
            if (this.sfrStroke && (div.strokeApplied == false))
            {
                this.sfrStroke.applyToElement(div);
                div.strokeApplied = true;
            }
            if (this.sfrReflection && (div.reflectionApplied == false))
            {
                this.sfrReflection.applyToElement(div);
                div.reflectionApplied = true;
            }
            if (this.sfrShadow && (!this.disableShadows) && (div.shadowApplied == false))
            {   
                this.sfrShadow.applyToElement(div);
                div.shadowApplied = true;
            }
        
            if (this.runningInApp && (window.webKitVersion <= 419) && this.preferences.setNeedsDisplay)
            {
                this.preferences.setNeedsDisplay();
            }
        }

        if (windowsInternetExplorer)
        {
            var cropDivs = div.select(".crop");
            var cropDiv = cropDivs[cropDivs.length - 1]; // last one, since reflections have two
            if (cropDiv)
            {
                // <rdar://problem/5146698> BREAK:Blog/Podcast: Excerpt image link doesn't work in IE6/7
                // The image in the cropping div doesn't behave like a hyperlink in IE. Add these behaviors to fake it.
                cropDiv.onclick = function()
                {
                    var anchorNode = div.parentNode;
                    var targetHref = locationHRef(); // default to this page if we can't find one
                    while (anchorNode && (anchorNode.tagName != "A"))
                    {
                        anchorNode = anchorNode.parentNode
                    }
                    if (anchorNode)
                    {
                        targetHref = anchorNode.href;
                    }
                    window.location = targetHref;
                };
                cropDiv.onmouseover = function()
                {
                    this.style.cursor = 'pointer';
                }
            }
        }
    },

    summaryExcerpt: function(descriptionHTML, maxSummaryLength)
    {
        var div = document.createElement("div");
        div.innerHTML = descriptionHTML;

        if (maxSummaryLength > 0)
        {
            var model = new HTMLTextModel(div);
            model.truncateAroundPosition(maxSummaryLength, "...");
        }
        else if (maxSummaryLength === 0)
        {
            div.innerHTML = "";
        }

        return div.innerHTML;
    }

});

var PrefMarkupWidget = Class.create(Widget, {

    initialize: function($super, instanceID, widgetPath, sharedPath, sitePath, preferences, runningInApp)
    {
        if (instanceID)
        {
            $super(instanceID, widgetPath, sharedPath, sitePath, preferences, runningInApp);
        }
    },

    onload: function()
    {
        if (!this.runningInApp)
        {
            this.setUpSubDocumentOnLoad();
        }
    },

    setUpSubDocumentOnLoad: function() 
    {
        var self = this;
        var oIFrame = this.getElementById("frame");

        // If we have iframe element in the widget, we need to wait until it is loaded completely.
        // Otherwise (in case of default image view) no special treatment is required.
        if (oIFrame)
        {
            // FireFox and Safari/WebKit don't support onload for an iframe.
            // IE does support iframe.onload, but it doesn't seem reliable 
            // -- perhaps the main document's onload is too late?
            // Polling is sadly the most reliable way.
            setTimeout(function() { self.loadedSubDocument() }, 250);

            // if we were using the iframe's onload, this would be it:
            // oIFrame.onload = function() { self.loadedSubDocument() };
        }
    },

    loadedSubDocument: function()
    {
        var oIFrame = this.getElementById("frame");
        var oSubDocument = oIFrame.contentWindow || oIFrame.contentDocument;
    
        // Get the subdocument through the DOM.  We need to do some 
        // manipulation on that document in some cases.
        if (oSubDocument.document)
        {
            oSubDocument = oSubDocument.document;
        }
    
        if (oSubDocument.body)
        {
            // The subdocument is properly loaded.  We can proceed.
        
            // Fix links and forms so they load in the top-level document, not this iframe.
            this.fixTargetOnElements(oSubDocument, "a");
            this.fixTargetOnElements(oSubDocument, "form");
        }
        else
        {
            // Try again in another 1/4 second
            var self = this;
            setTimeout(function() { self.loadedSubDocument() }, 250);
        }
    },

    fixTargetOnElements: function(doc, tagName)
    {
        var elements = doc.getElementsByTagName(tagName);
        for (var i=0; i < elements.length; i++)
        {
            var target = elements[i].target;
            if( target === undefined || target == "" )
                elements[i].target = "_top";            
        }    
    }
});

//>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

//<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
/*
© Copyright 2007 Apple Inc. All rights reserved.

IMPORTANT:  This Apple software and the associated images located in
/System/Library/WidgetResources/AppleClasses/ (collectively "Apple Software")
are supplied to you by Apple Computer, Inc. (“Apple”) in consideration of your
agreement to the following terms. Your use, installation and/or redistribution
of this Apple Software constitutes acceptance of these terms. If you do not
agree with these terms, please do not use, install, or redistribute this Apple
Software.

In consideration of your agreement to abide by the following terms, and subject
to these terms, Apple grants you a personal, non-exclusive license, under
Apple’s copyrights in the Apple Software, to use, reproduce, and redistribute
the Apple Software, in text form (for JavaScript files) or binary form (for
associated images), for the sole purpose of creating Dashboard widgets for Mac
OS X.

If you redistribute the Apple Software, you must retain this entire notice and
the warranty disclaimers and limitation of liability provisions (last two
paragraphs below) in all such redistributions of the Apple Software.

You may not use the name, trademarks, service marks or logos of Apple to endorse
or promote products that include the Apple Software without the prior written
permission of Apple. Except as expressly stated in this notice, no other rights
or licenses, express or implied, are granted by Apple herein, including but not
limited to any patent rights that may be infringed by your products that
incorporate the Apple Software or by other works in which the Apple Software may
be incorporated.

The Apple Software is provided on an "AS IS" basis.  APPLE MAKES NO WARRANTIES,
EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF
NON-INFRINGEMENT, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE,
REGARDING THE APPPLE SOFTWARE OR ITS USE AND OPERATION ALONE OR IN COMBINATION
WITH YOUR PRODUCTS.

IN NO EVENT SHALL APPLE BE LIABLE FOR ANY SPECIAL, INDIRECT, INCIDENTAL OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
ARISING IN ANY WAY OUT OF THE USE, REPRODUCTION, AND/OR DISTRIBUTION OF THE
APPLE SOFTWARE, HOWEVER CAUSED AND WHETHER UNDER THEORY OF CONTRACT, TORT
(INCLUDING NEGLIGENCE), STRICT LIABILITY OR OTHERWISE, EVEN IF APPLE HAS BEEN
ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

function IWScrollbar(scrollbar)
{
}

/*
 * _init() member function
 * Initialize the scrollbar.
 * pre: this.scrollbar
 * post: this._thumb, this._track + event handlers
 */
IWScrollbar.prototype._init = function()
{
    var style = null;
    var element = null;
        
    // Scrollbar Track
    this._track = $(document.createElement("div"));
    style = this._track.style;
    // fill our containing div
    style.height = "100%";
    style.width = "100%";
    this.scrollbar.appendChild(this._track);
    
    // Scrollbar Track Top
    element = $(document.createElement("div"));
    element.style.position = "absolute";
    this._setObjectStart(element, 0);
    this._track.appendChild(element);
    
    // Scrollbar Track Middle
    element = $(document.createElement("div"));
    element.style.position = "absolute";
    this._track.appendChild(element);
    
    // Scrollbar Track Bottom
    element = $(document.createElement("div"));
    element.style.position = "absolute";
    windowsInternetExplorer || this._setObjectEnd(element, 0);
    this._track.appendChild(element);
    
    // Scrollbar Thumb
    this._thumb = $(document.createElement("div"));
    style = this._thumb.style;
    style.position = "absolute";
    this._setObjectSize(this._thumb, this.minThumbSize); // default size
    this._track.appendChild(this._thumb);
    
    // Scrollbar Thumb Top
    element = $(document.createElement("div"));
    element.style.position = "absolute";
    this._setObjectStart(element, 0);
    this._thumb.appendChild(element);
    
    // Scrollbar Thumb Middle
    element = $(document.createElement("div"));
    element.style.position = "absolute";
    this._thumb.appendChild(element);
    
    // Scrollbar Thumb Bottom
    element = $(document.createElement("div"));
    element.style.position = "absolute";
    windowsInternetExplorer || this._setObjectEnd(element, 0);
    this._thumb.appendChild(element);
    
    // Set up styles
    this.setSize(this.size);
    
    this.setTrackStart(this.trackStartPath, this.trackStartLength);
    this.setTrackMiddle(this.trackMiddlePath);
    this.setTrackEnd(this.trackEndPath, this.trackEndLength);
    this.setThumbStart(this.thumbStartPath, this.thumbStartLength);
    this.setThumbMiddle(this.thumbMiddlePath);
    this.setThumbEnd(this.thumbEndPath, this.thumbEndLength);
    
    // hide the thumb until we refresh
    this._thumb.style.display = "none";
    
    // Add event listeners
    Event.observe(this._track, "mousedown", this._mousedownTrackHandler, false);
    Event.observe(this._thumb, "mousedown", this._mousedownThumbHandler, false);
    // ScrollArea will fire a refresh for us
}

IWScrollbar.prototype.remove = function()
{
    this.scrollbar.removeChild(this._track);
}

// Capture events that we don't handle but also don't want getting through
IWScrollbar.prototype._captureEvent = function(event)
{
    event.stopPropagation();
    event.preventDefault();
}

/*********************
 * Thumb scroll events
 */
IWScrollbar.prototype._mousedownThumb = function(event)
{
    // temporary event listeners
    Event.observe(document, "mousemove", this._mousemoveThumbHandler, true);
    Event.observe(document, "mouseup",   this._mouseupThumbHandler, true);
    Event.observe(document, "mouseover", this._captureEventHandler, true);
    Event.observe(document, "mouseout",  this._captureEventHandler, true);
    
    this._thumbStart_temp = this._getMousePosition(event);
    
    this._scroll_thumbStartPos = this._getThumbStartPos();
    
    event.stopPropagation();
    event.preventDefault();
}

IWScrollbar.prototype._mousemoveThumb = function(event)
{
    var delta = this._getMousePosition(event) - this._thumbStart_temp;
    
    var new_pos = this._scroll_thumbStartPos + delta;
    this.scrollTo(this._contentPositionForThumbPosition(new_pos));
    
    event.stopPropagation();
    event.preventDefault();
}

IWScrollbar.prototype._mouseupThumb = function(event)
{
    // remove temporary event handlers
    Event.stopObserving(document, "mousemove", this._mousemoveThumbHandler, true);
    Event.stopObserving(document, "mouseup", this._mouseupThumbHandler, true);
    Event.stopObserving(document, "mouseover", this._captureEventHandler, true);
    Event.stopObserving(document, "mouseout", this._captureEventHandler, true);
    
    // remove temporary properties
    delete this._thumbStart_temp;
    delete this._scroll_thumbStartPos;
    
    event.stopPropagation();
    event.preventDefault();
}

/*********************
 * Track scroll events
 */
IWScrollbar.prototype._mousedownTrack = function(event)
{
    this._track_mouse_temp = this._getMousePosition(event) - this._trackOffset;
    
    if (event.altKey)
    {
        this.scrollTo(this._contentPositionForThumbPosition(this._track_mouse_temp - (this._thumbLength / 2)));
        delete this._track_mouse_temp;
    }
    else
    {
        this._track_scrolling = true;
        
        // temporary event handlers
        Event.observe(this._track, "mousemove", this._mousemoveTrackHandler, true);
        Event.observe(this._track, "mouseover", this._mouseoverTrackHandler, true);
        Event.observe(this._track, "mouseout", this._mouseoutTrackHandler, true);
        Event.observe(document,    "mouseup", this._mouseupTrackHandler, true);
        
        this._trackScrollOnePage(this);
        this._track_timer = setInterval(this._trackScrollDelay, 500, this);
    }
    
    event.stopPropagation();
    event.preventDefault();
}

IWScrollbar.prototype._trackScrollDelay = function(self)
{
    if (!self._track_scrolling) return;
    
    clearInterval(self._track_timer);
    
    self._trackScrollOnePage(self);
    self._track_timer = setInterval(self._trackScrollOnePage, 150, self);
}

IWScrollbar.prototype._mousemoveTrack = function(event)
{
    this._track_mouse_temp = this._getMousePosition(event) - this._trackOffset;
    
    event.stopPropagation();
    event.preventDefault();
}

IWScrollbar.prototype._mouseoverTrack = function(event)
{
    this._track_mouse_temp = this._getMousePosition(event) - this._trackOffset;
    this._track_scrolling = true;
    
    event.stopPropagation();
    event.preventDefault();
}

IWScrollbar.prototype._mouseoutTrack = function(event)
{
    this._track_scrolling = false;
    
    event.stopPropagation();
    event.preventDefault();
}

IWScrollbar.prototype._mouseupTrack = function(event)
{
    clearInterval(this._track_timer);
    
    // clear temporary event handlers
    Event.stopObserving(this._track, "mousemove", this._mousemoveTrackHandler, true);
    Event.stopObserving(this._track, "mouseover", this._mouseoverTrackHandler, true);
    Event.stopObserving(this._track, "mouseout", this._mouseoutTrackHandler, true);
    Event.stopObserving(document,    "mouseup", this._mouseupTrackHandler, true);
    
    // remove temporary properties
    delete this._track_mouse_temp;
    delete this._track_scrolling;
    delete this._track_timer;
    
    event.stopPropagation();
    event.preventDefault();
}

IWScrollbar.prototype._trackScrollOnePage = function(self)
{
    // this is called from setInterval, so we need a ptr to this
    
    if (!self._track_scrolling) return;
    
    var deltaScroll = Math.round(self._trackLength * self._getViewToContentRatio());
    
    if (self._track_mouse_temp < self._thumbStart)
        self.scrollByThumbDelta(-deltaScroll);
    else if (self._track_mouse_temp > (self._thumbStart + self._thumbLength))
        self.scrollByThumbDelta(deltaScroll);
}

/*
 * setScrollArea(scrollarea)
 * Sets the IWScrollArea this scrollbar is using.
 */
IWScrollbar.prototype.setScrollArea = function(scrollarea)
{
    // if we already have a scrollarea, remove the mousewheel handler
    if (this.scrollarea)
    {
        Event.stopObserving(this.scrollbar, "mousewheel", this.scrollarea._mousewheelScrollHandler, true);     // Safari/IE
        Event.stopObserving(this.scrollbar, "DOMMouseScroll", this.scrollarea._mousewheelScrollHandler, true); // Firefox
    }
    
    this.scrollarea = scrollarea;
    
    // add mousewheel handler
    Event.observe(this.scrollbar, "mousewheel",     this.scrollarea._mousewheelScrollHandler, true); // Safari/IE
    Event.observe(this.scrollbar, "DOMMouseScroll", this.scrollarea._mousewheelScrollHandler, true); // Firefox
}

/*
 * refresh()
 * Refresh the scrollbar size and thumb position
 */
IWScrollbar.prototype.refresh = function()
{
    this._trackOffset = this._computeTrackOffset();
    this._trackLength = this._computeTrackLength();
    
    var ratio = this._getViewToContentRatio();
    
    if (ratio >= 1.0 || !this._canScroll())
    {
        if (this.autohide)
        {
            // hide the scrollbar, all content is visible
            this.hide();
        }
        
        // hide the thumb
        this._thumb.style.display = "none";
        this.scrollbar.style.appleDashboardRegion = "none";
    }
    else
    {
        this._thumbLength = Math.max(Math.round(this._trackLength * ratio), this.minThumbSize);
        this._numScrollablePixels = this._trackLength - this._thumbLength - (2 * this.padding);
        
        this._setObjectLength(this._thumb, this._thumbLength);

        if (windowsInternetExplorer)
        {
            this._setObjectStart(this._thumb.down().next(), this.thumbStartLength);
            this._setObjectLength(this._thumb.down().next(), this._thumbLength
                - this.thumbStartLength - this.thumbEndLength);

            this._setObjectStart(this._thumb.down().next(1), this._thumbLength - this.thumbEndLength);
            this._setObjectLength(this._thumb.down().next(1), this.thumbEndLength);

            if (!this.fixedUpIEPNGBGs)
            {
                fixupIEPNGBGsInTree(this._track);

                // Must reinstall the mousedown handlers after the PNG BG fix.  The new child 
                // elements don't seem to bubble events properly, possibly because they were 
                // created after the handlers were installed
                Event.stopObserving(this._track, "mousedown", this._mousedownTrackHandler);
                Event.stopObserving(this._thumb, "mousedown", this._mousedownThumbHandler);
                Event.observe(this._track, "mousedown", this._mousedownTrackHandler);
                Event.observe(this._thumb, "mousedown", this._mousedownThumbHandler);
                    
                this.fixedUpIEPNGBGs = true;
            }

        }
                
        // show the thumb
        this._thumb.style.display = "block";
        this.scrollbar.style.appleDashboardRegion = "dashboard-region(control rectangle)";
        
        this.show();
    }
    
    // Make sure position is updated appropriately
    this.verticalHasScrolled();
    this.horizontalHasScrolled();
}

IWScrollbar.prototype.setAutohide = function(autohide)
{
    this.autohide = autohide;
    
    // hide the scrollbar if necessary
    if (this._getViewToContentRatio() >= 1.0 && autohide)
    {
        this.hide();
    }
    else
    {
        this.show();
    }
}

IWScrollbar.prototype.hide = function()
{
    this._track.style.display = "none";
    this.hidden = true;
}

IWScrollbar.prototype.show = function()
{
    this._track.style.display = "block";
    this.hidden = false;
}

IWScrollbar.prototype.setSize = function(size)
{
    this.size = size;
    
    this._setObjectSize(this.scrollbar, size);
    this._setObjectSize(this._track.down().next(), size);
    this._setObjectSize(this._thumb.down().next(), size);
}

IWScrollbar.prototype.setTrackStart = function(imgpath, length)
{
    this.trackStartPath = imgpath;
    this.trackStartLength = length;

    var element = this._track.down();
    element.style.background = "url(" + imgpath + ") no-repeat top left";
    this._setObjectLength(element, length);
    this._setObjectSize(element, this.size);
    this._setObjectStart(this._track.down().next(), length);
}

IWScrollbar.prototype.setTrackMiddle = function(imgpath)
{
    this.trackMiddlePath = imgpath;

    this._track.down().next().style.background = "url(" + imgpath + ") " + this._repeatType + " top left";
}

IWScrollbar.prototype.setTrackEnd = function(imgpath, length)
{
    this.trackEndPath = imgpath;
    this.trackEndLength = length;

    var element = this._track.down().next(1);
    element.style.background = "url(" + imgpath + ") no-repeat top left";
    this._setObjectLength(element, length);
    this._setObjectSize(element, this.size);
    windowsInternetExplorer || this._setObjectEnd(this._track.down().next(), length);
}

IWScrollbar.prototype.setThumbStart = function(imgpath, length)
{
    this.thumbStartPath = imgpath;
    this.thumbStartLength = length;
    
    var element = this._thumb.down();
    element.style.background = "url(" + imgpath + ") no-repeat top left";
    this._setObjectLength(element, length);
    this._setObjectSize(element, this.size);
    this._setObjectStart(this._thumb.down().next(), length);
}

IWScrollbar.prototype.setThumbMiddle = function(imgpath)
{
    this.thumbMiddlePath = imgpath;
    
    this._thumb.down().next().style.background = "url(" + imgpath + ") " + this._repeatType + " top left";
}

IWScrollbar.prototype.setThumbEnd = function(imgpath, length)
{
    this.thumbEndPath = imgpath;
    this.thumbEndLength = length;

    var element = this._thumb.down().next(1);
    element.style.background = "url(" + imgpath + ") no-repeat top left";
    this._setObjectLength(element, length);
    this._setObjectSize(element, this.size);
    windowsInternetExplorer || this._setObjectEnd(this._thumb.down().next(), length);
}

IWScrollbar.prototype._contentPositionForThumbPosition = function(thumb_pos)
{
    // if we're currently displaying all content, we don't want it outside the view
    if (this._getViewToContentRatio() >= 1.0)
    {
        return 0;
    }
    else
    {
        return (thumb_pos - this.padding) * ((this._getContentLength() - this._getViewLength()) / this._numScrollablePixels);
    }
}

IWScrollbar.prototype._thumbPositionForContentPosition = function(page_pos)
{
    // if we're currently displaying all content, we don't want it outside the view
    if (this._getViewToContentRatio() >= 1.0)
    {
        return this.padding;
    }
    else
    {
        var result = this.padding + (page_pos / ((this._getContentLength() - this._getViewLength()) / this._numScrollablePixels));
        if (isNaN(result))
            result = 0;
        return result;
    }
}

IWScrollbar.prototype.scrollByThumbDelta = function(deltaScroll)
{
    if (deltaScroll == 0)
        return;
    
    this.scrollTo(this._contentPositionForThumbPosition(this._thumbStart + deltaScroll));
}


/*******************************************************************************
 * IWVerticalScrollbar
 * Implementation of IWScrollbar
 *
 *
 */

function IWVerticalScrollbar(scrollbar)
{
    /* Objects */
    this.scrollarea = null;
    this.scrollbar = $(scrollbar);
    
    /* public properties */
    // These are read-write. Set them as needed.
    this.minThumbSize = 28;
    this.padding = -1;
    
    // These are read-only. Use the setter functions to set them.
    this.autohide = true;
    this.hidden = true;
    this.size = 19; // width
    this.trackStartPath = transparentGifURL();
    this.trackStartLength = 18; // height
    this.trackMiddlePath = transparentGifURL();
    this.trackEndPath = transparentGifURL();
    this.trackEndLength = 18; // height
    this.thumbStartPath = transparentGifURL();
    this.thumbStartLength = 9; // height
    this.thumbMiddlePath = transparentGifURL();
    this.thumbEndPath = transparentGifURL();
    this.thumbEndLength = 9; // height

    /* Internal objects */
    this._track = null;
    this._thumb = null;
    
    /* Dimensions */
    // these only need to be set during refresh()
    this._trackOffset = 0;
    this._trackLength = 0;
    this._numScrollablePixels = 0;
    this._thumbLength = 0;
    this._repeatType = "repeat-y";
    
    // these change as the content is scrolled
    this._thumbStart = this.padding;
    
    // For JavaScript event handlers
    var _self = this;
    
    this._captureEventHandler = function(event) { _self._captureEvent(event); };
    this._mousedownThumbHandler = function(event) { _self._mousedownThumb(event); };
    this._mousemoveThumbHandler = function(event) { _self._mousemoveThumb(event); };
    this._mouseupThumbHandler = function(event) { _self._mouseupThumb(event); };
    this._mousedownTrackHandler = function(event) { _self._mousedownTrack(event); };
    this._mousemoveTrackHandler = function(event) { _self._mousemoveTrack(event); };
    this._mouseoverTrackHandler = function(event) { _self._mouseoverTrack(event); };
    this._mouseoutTrackHandler = function(event) { _self._mouseoutTrack(event); };
    this._mouseupTrackHandler = function(event) { _self._mouseupTrack(event); };
    
    this._init();
}

// Inherit from IWScrollbar
IWVerticalScrollbar.prototype = new IWScrollbar(null);


/*********************
 * Orientation-specific functions.
 * These helper functions return vertical values.
 */
IWVerticalScrollbar.prototype.scrollTo = function(pos)
{
    this.scrollarea.verticalScrollTo(pos);
}

IWVerticalScrollbar.prototype._setObjectSize = function(object, size)
{ object.style.width = size + "px"; }

IWVerticalScrollbar.prototype._setObjectLength = function(object, length)
{ object.style.height = length + "px"; }

IWVerticalScrollbar.prototype._setObjectStart = function(object, start)
{ object.style.top = start + "px"; }

IWVerticalScrollbar.prototype._setObjectEnd = function(object, end)
{ object.style.bottom = end + "px"; }

IWVerticalScrollbar.prototype._getMousePosition = function(event)
{
    if (event != undefined)
        return Event.pointerY(event);
    else
        return 0;
}
    
IWVerticalScrollbar.prototype._getThumbStartPos = function()
{
    return this._thumb.offsetTop;
}

IWVerticalScrollbar.prototype._computeTrackOffset = function()
{
    // get the absolute top of the track
    var obj = this.scrollbar;
    var curtop = 0;
    while (obj.offsetParent)
    {
        curtop += obj.offsetTop;
        obj = obj.offsetParent;
    }
    
    return curtop;
}

IWVerticalScrollbar.prototype._computeTrackLength = function()
{
    // get the current actual track height
    return this.scrollbar.offsetHeight;
}

IWVerticalScrollbar.prototype._getViewToContentRatio = function()
{ return this.scrollarea.viewToContentHeightRatio; }

IWVerticalScrollbar.prototype._getContentLength = function()
{ return this.scrollarea.content.scrollHeight; }

IWVerticalScrollbar.prototype._getViewLength = function()
{ return this.scrollarea.viewHeight; }

IWVerticalScrollbar.prototype._canScroll = function()
{ return this.scrollarea.scrollsVertically; }


IWVerticalScrollbar.prototype.verticalHasScrolled = function()
{
    var new_thumb_pos = this._thumbPositionForContentPosition(this.scrollarea.content.scrollTop);
    this._thumbStart = new_thumb_pos;
    this._thumb.style.top = new_thumb_pos + "px";
}

IWVerticalScrollbar.prototype.horizontalHasScrolled = function()
{
}



/*******************************************************************************
* IWHorizontalScrollbar
* Implementation of IWScrollbar
*
*
*/

function IWHorizontalScrollbar(scrollbar)
{
    /* Objects */
    this.scrollarea = null;
    this.scrollbar = $(scrollbar);
    
    /* public properties */
    // These are read-write. Set them as needed.
    this.minThumbSize = 28;
    this.padding = -1;
    
    // These are read-only. Use the setter functions to set them.
    this.autohide = true;
    this.hidden = true;
    this.size = 19; // height
    this.trackStartPath = transparentGifURL();
    this.trackStartLength = 18; // width
    this.trackMiddlePath = transparentGifURL();
    this.trackEndPath = transparentGifURL();
    this.trackEndLength = 18; // width
    this.thumbStartPath = transparentGifURL();
    this.thumbStartLength = 9; // width
    this.thumbMiddlePath = transparentGifURL();
    this.thumbEndPath = transparentGifURL();
    this.thumbEndLength = 9; // width
    
    /* Internal objects */
    this._track = null;
    this._thumb = null;
    
    /* Dimensions */
    // these only need to be set during refresh()
    this._trackOffset = 0;
    this._trackLength = 0;
    this._numScrollablePixels = 0;
    this._thumbLength = 0;
    this._repeatType = "repeat-x";
    
    // these change as the content is scrolled
    this._thumbStart = this.padding;
    
    // For JavaScript event handlers
    var _self = this;
    
    this._captureEventHandler = function(event) { _self._captureEvent(event); };
    this._mousedownThumbHandler = function(event) { _self._mousedownThumb(event); };
    this._mousemoveThumbHandler = function(event) { _self._mousemoveThumb(event); };
    this._mouseupThumbHandler = function(event) { _self._mouseupThumb(event); };
    this._mousedownTrackHandler = function(event) { _self._mousedownTrack(event); };
    this._mousemoveTrackHandler = function(event) { _self._mousemoveTrack(event); };
    this._mouseoverTrackHandler = function(event) { _self._mouseoverTrack(event); };
    this._mouseoutTrackHandler = function(event) { _self._mouseoutTrack(event); };
    this._mouseupTrackHandler = function(event) { _self._mouseupTrack(event); };
    
    this._init();
}

// Inherit from IWScrollbar
IWHorizontalScrollbar.prototype = new IWScrollbar(null);


/*********************
* Orientation-specific functions.
* These helper functions return vertical values.
*/
IWHorizontalScrollbar.prototype.scrollTo = function(pos)
{
    this.scrollarea.horizontalScrollTo(pos);
}

IWHorizontalScrollbar.prototype._setObjectSize = function(object, size)
{ object.style.height = size + "px"; }

IWHorizontalScrollbar.prototype._setObjectLength = function(object, length)
{ object.style.width = length + "px"; }

IWHorizontalScrollbar.prototype._setObjectStart = function(object, start)
{ object.style.left = start + "px"; }

IWHorizontalScrollbar.prototype._setObjectEnd = function(object, end)
{ object.style.right = end + "px"; }

IWHorizontalScrollbar.prototype._getMousePosition = function(event)
{
    if (event != undefined)
        return Event.pointerX(event);
    else
        return 0;
}

IWHorizontalScrollbar.prototype._getThumbStartPos = function()
{
    return this._thumb.offsetLeft;
}

IWHorizontalScrollbar.prototype._computeTrackOffset = function()
{
    // get the absolute top of the track
    var obj = this.scrollbar;
    var curtop = 0;
    while (obj.offsetParent)
    {
        curtop += obj.offsetLeft;
        obj = obj.offsetParent;
    }
    
    return curtop;
}

IWHorizontalScrollbar.prototype._computeTrackLength = function()
{
    // get the current actual track height
    return this.scrollbar.offsetWidth;
}

IWHorizontalScrollbar.prototype._getViewToContentRatio = function()
{ return this.scrollarea.viewToContentWidthRatio; }

IWHorizontalScrollbar.prototype._getContentLength = function()
{ return this.scrollarea.content.scrollWidth; }

IWHorizontalScrollbar.prototype._getViewLength = function()
{ return this.scrollarea.viewWidth; }

IWHorizontalScrollbar.prototype._canScroll = function()
{ return this.scrollarea.scrollsHorizontally; }


IWHorizontalScrollbar.prototype.verticalHasScrolled = function()
{
}

IWHorizontalScrollbar.prototype.horizontalHasScrolled = function()
{
    var new_thumb_pos = this._thumbPositionForContentPosition(this.scrollarea.content.scrollLeft);
    this._thumbStart = new_thumb_pos;
    this._thumb.style.left = new_thumb_pos + "px";
}

//>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

//<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
/*
© Copyright 2007 Apple Inc. All rights reserved.

IMPORTANT:  This Apple software and the associated images located in
/System/Library/WidgetResources/AppleClasses/ (collectively "Apple Software")
are supplied to you by Apple Computer, Inc. (“Apple”) in consideration of your
agreement to the following terms. Your use, installation and/or redistribution
of this Apple Software constitutes acceptance of these terms. If you do not
agree with these terms, please do not use, install, or redistribute this Apple
Software.

In consideration of your agreement to abide by the following terms, and subject
to these terms, Apple grants you a personal, non-exclusive license, under
Apple’s copyrights in the Apple Software, to use, reproduce, and redistribute
the Apple Software, in text form (for JavaScript files) or binary form (for
associated images), for the sole purpose of creating Dashboard widgets for Mac
OS X.

If you redistribute the Apple Software, you must retain this entire notice and
the warranty disclaimers and limitation of liability provisions (last two
paragraphs below) in all such redistributions of the Apple Software.

You may not use the name, trademarks, service marks or logos of Apple to endorse
or promote products that include the Apple Software without the prior written
permission of Apple. Except as expressly stated in this notice, no other rights
or licenses, express or implied, are granted by Apple herein, including but not
limited to any patent rights that may be infringed by your products that
incorporate the Apple Software or by other works in which the Apple Software may
be incorporated.

The Apple Software is provided on an "AS IS" basis.  APPLE MAKES NO WARRANTIES,
EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF
NON-INFRINGEMENT, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE,
REGARDING THE APPPLE SOFTWARE OR ITS USE AND OPERATION ALONE OR IN COMBINATION
WITH YOUR PRODUCTS.

IN NO EVENT SHALL APPLE BE LIABLE FOR ANY SPECIAL, INDIRECT, INCIDENTAL OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
ARISING IN ANY WAY OUT OF THE USE, REPRODUCTION, AND/OR DISTRIBUTION OF THE
APPLE SOFTWARE, HOWEVER CAUSED AND WHETHER UNDER THEORY OF CONTRACT, TORT
(INCLUDING NEGLIGENCE), STRICT LIABILITY OR OTHERWISE, EVEN IF APPLE HAS BEEN
ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * IWScrollArea Constructor
 * content is the element containing the display to be scrolled.
 * Any additional arguments will be added as scrollbars using this.addScrollbar.
 */
function IWScrollArea(content)
{
    /* Objects */
    this.content = $(content);
    
    /* public properties */
    // These are read-write. Set them as needed.
    this.scrollsVertically = true;
    this.scrollsHorizontally = true;
    this.singlepressScrollPixels = 10;
    
    // These are read-only.
    this.viewHeight = 0;
    this.viewToContentHeightRatio = 1.0;
    this.viewWidth = 0;
    this.viewToContentWidthRatio = 1.0;

    /* Internal Objects */
    this._scrollbars = new Array();
    
    // For JavaScript event handlers
    var _self = this;
    
    /*
     * Privileged methods
     * These event handlers need to be here because within an event handler,
     * "this" refers to the element which called the event, rather than the 
     * class instance.
     */ 
    this._refreshHandler = function() { _self.refresh(); };
    this._keyPressedHandler = function() { _self.keyPressed(event); };
    this._mousewheelScrollHandler = function(event) { _self.mousewheelScroll(event); };

    // Set up the style for the content element just to be certain
    this.content.style.overflow = "hidden";
    this.content.scrollTop = 0;
    this.content.scrollLeft = 0;
    
    // Add event listeners
    Event.observe(this.content, "mousewheel",     this._mousewheelScrollHandler, true); // Safari/IE
    Event.observe(this.content, "DOMMouseScroll", this._mousewheelScrollHandler, true); // Firefox
    
    this.refresh();
    
    // Add any scrollbars
    var c = arguments.length;
    for (var i = 1; i < c; ++i)
    {
        this.addScrollbar(arguments[i]);
    }
}

IWScrollArea.prototype.addScrollbar = function(scrollbar)
{
    scrollbar.setScrollArea(this);
    this._scrollbars.push(scrollbar);
    scrollbar.refresh();
}

IWScrollArea.prototype.removeScrollbar = function(scrollbar)
{
    var scrollbars = this._scrollbars;
    var c = scrollbars.length;
    for (var i = 0; i < c; ++i)
    {
        if (scrollbars[i] == scrollbar)
        {
            scrollbars.splice(i, 1);
            break;
        }
    }
}

IWScrollArea.prototype.remove = function()
{
    Event.stopObserving(this.content, "mousewheel",     this._mousewheelScrollHandler, true); // Safari/IE
    Event.stopObserving(this.content, "DOMMouseScroll", this._mousewheelScrollHandler, true); // Firefox
    
    var scrollbars = this._scrollbars;
    var c = scrollbars.length;
    for (var i = 0; i < c; ++i)
    {
        scrollbars[i].setScrollArea(null);
    }
}

/*
 * refresh() member function
 * Refresh the current scrollbar position and size.
 * This should be called whenever the content element changes.
 */
IWScrollArea.prototype.refresh = function()
{   
    // get the current actual view height. Float because we divide.
    this.viewHeight = this.content.offsetHeight;
    this.viewWidth  = this.content.offsetWidth;
    
    if (this.content.scrollHeight > this.viewHeight)
    {
        this.viewToContentHeightRatio = this.viewHeight / this.content.scrollHeight;
        this.verticalScrollTo(this.content.scrollTop);
    }
    else
    {
        this.viewToContentHeightRatio = 1.0;
        this.verticalScrollTo(0);
    }
    
    if (this.content.scrollWidth > this.viewWidth)
    {
        this.viewToContentWidthRatio = this.viewWidth / this.content.scrollWidth;
        this.horizontalScrollTo(this.content.scrollLeft);
    }
    else
    {
        this.viewToContentWidthRatio = 1.0;
        this.horizontalScrollTo(0);
    }
    
    var scrollbars = this._scrollbars;
    var c = scrollbars.length;
    for (var i = 0; i < c; ++i)
    {
        scrollbars[i].refresh();
    }
}

/*
 * focus() member function.
 * Tell the scrollarea that it is in focus. It will capture keyPressed events
 * and if they are arrow keys scroll accordingly.
 */
IWScrollArea.prototype.focus = function()
{
    Event.observe(document, "keypress", this._keyPressedHandler, true);
}

/*
 * blur() member function.
 * Tell the scrollarea that it is no longer in focus. It will cease capturing
 * keypress events.
 */
IWScrollArea.prototype.blur = function()
{
    Event.stopObserving(document, "keypress", this._keyPressedHandler, true);
}


/*
 * reveal(element) member function.
 * Pass in an Element which is contained within the content element.
 * The content will then be scrolled to reveal that element.
 */
IWScrollArea.prototype.reveal = function(element)
{
    var offsetY = 0;
    var obj = element;
    do
    {
        offsetY += obj.offsetTop;
        obj = obj.offsetParent;
    } while (obj && obj != this.content);
    
    var offsetX = 0;
    obj = element;
    do
    {
        offsetX += obj.offsetLeft;
        obj = obj.offsetParent;
    } while (obj && obj != this.content);
    
    this.verticalScrollTo(offsetY);
    this.horizontalScrollTo(offsetX);
}


IWScrollArea.prototype.verticalScrollTo = function(new_content_top)
{
    if (!this.scrollsVertically)
        return;
    
    var bottom = this.content.scrollHeight - this.viewHeight;
    
    if (new_content_top < 0)
    {
        new_content_top = 0;
    }
    else if (new_content_top > bottom)
    {
        new_content_top = bottom;
    }
    
    this.content.scrollTop = new_content_top;
    
    var scrollbars = this._scrollbars;
    var c = scrollbars.length;
    for (var i = 0; i < c; ++i)
    {
        scrollbars[i].verticalHasScrolled();
    }
}

IWScrollArea.prototype.horizontalScrollTo = function(new_content_left)
{
    if (!this.scrollsHorizontally)
        return;
    
    var right = this.content_width - this.viewWidth;
    
    if (new_content_left < 0)
    {
        new_content_left = 0;
    }
    else if (new_content_left > right)
    {
        new_content_left = right;
    }
    
    this.content.scrollLeft = new_content_left;
    
    var scrollbars = this._scrollbars;
    var c = scrollbars.length;
    for (var i = 0; i < c; ++i)
    {
        scrollbars[i].horizontalHasScrolled();
    }
}

/*********************
 * Keypressed events
 */
IWScrollArea.prototype.keyPressed = function(event)
{
    var handled = true;
    
    if (event.altKey)
        return;
    if (event.shiftKey)
        return;
    
    switch (event.keyIdentifier)
    {
        case "Home":
            this.verticalScrollTo(0);
            break;
        case "End":
            this.verticalScrollTo(this.content.scrollHeight - this.viewHeight);
            break;
        case "Up":
            this.verticalScrollTo(this.content.scrollTop - this.singlepressScrollPixels);
            break;
        case "Down":
            this.verticalScrollTo(this.content.scrollTop + this.singlepressScrollPixels);
            break;
        case "PageUp":
            this.verticalScrollTo(this.content.scrollTop - this.viewHeight);
            break;
        case "PageDown":
            this.verticalScrollTo(this.content.scrollTop + this.viewHeight);
            break;
        case "Left":
            this.horizontalScrollTo(this.content.scrollLeft - this.singlepressScrollPixels);
            break;
        case "Right":
            this.horizontalScrollTo(this.content.scrollLeft + this.singlepressScrollPixels);
            break;
        default:
            handled = false;
    }
    
    if (handled)
    {
        event.stopPropagation();
        event.preventDefault();
    }
}

/*********************
 * Scrollwheel events
 */
IWScrollArea.prototype.mousewheelScroll = function(event)
{
    var deltaScroll = event.wheelDelta ? 
        (event.wheelDelta / 120 * this.singlepressScrollPixels) : // Safari/IE
        (event.detail     / -2  * this.singlepressScrollPixels);  // Firefox
    this.verticalScrollTo(this.content.scrollTop - deltaScroll);

    event.stopPropagation();
    event.preventDefault();
}

// This is a semi-abstract Javascript class that represents a conceptual "view".
// It knows how to create a div element for itself, plus other standard stuff like
// how to redraw and resize.
var View = Class.create({

    initialize: function(widget, parentDiv)
    {
        this.m_widget = widget;
        this.m_parentDiv = parentDiv;
        this.m_divInstanceId = this.m_divId;
        this.hide();
    },

    ensureDiv: function()
    {
        var div = this.m_widget.div(this.m_divInstanceId);
        if (!div)
        {
            // Must use DOM; manipulating innerHTML on the parentDiv destroys and then
            // recreates its child DOM nodes, which messes up the DOM references 
            // in the scroll bar.
            div = new Element('div', {
                'id': this.m_widget.getInstanceId(this.m_divInstanceId)});

            div.addClassName(this.m_divClass);
            this.m_parentDiv.appendChild(div);
        }
        return $(div);
    },

    hide: function()
    {
        this.ensureDiv().hide();
    },

    show: function()
    {
        this.ensureDiv().show();
    },

    render: function()
    {
        // default does nothing
    },

    resize: function()
    {
        // default does nothing
    }
});

var StatusView = Class.create(View, {

    initialize: function($super, widget, parentDiv)
    {
        $super(widget, parentDiv);
        this.render();
        this.hide(); // Load images, but hide until requested
    },

    render: function()
    {
        // Use a table so that we can get vertical-align to work as we expect
        var markup = "<table class='StatusMessageTable'><tr><td>";

        if (this.badgeImage)
        {
            markup += imgMarkup(this.m_widget.widgetPath + "/" + this.badgeImage,
                "", 
                "id='" + this.p_badgeImgId() + "'", 
                "");
        }

        // Close centering table off
        markup += "</td></tr></table>";

        if (this.upperRightBadgeWidth && this.upperRightBadgeHeight)
        {
            var badgeURL = (this.upperRightBadge) ? 
                (this.m_widget.widgetPath + "/" + this.upperRightBadge) : 
                transparentGifURL();
            markup += imgMarkup(badgeURL, 
                "", 
                "class='StatusUpperRightBadge' width='" + this.upperRightBadgeWidth + 
                    "' height='" + this.upperRightBadgeHeight + "' ", 
                "");
        }

        // The translucent overlay in the background
        var overlayPath = this.m_widget.sharedPath.stringByAppendingPathComponent("Translucent-Overlay.png");
        markup += imgMarkup(overlayPath,
            "position: absolute; top: 0; left: 0;", 
            "id='" + this.p_overlayImgId() + "' width='700' height='286' ", 
            "");
            
        if (this.statusMessageKey)
        {
            markup += "<div id='" + this.p_statusMessageBlockId() + "' class='StatusMessageBlock' ><span>" +
                this.m_widget.localizedString(this.statusMessageKey) + 
                "</span></div>";
        }

        this.ensureDiv().update(markup);
        this.resize();
    },

    resize: function()
    {
        var widgetWidth = (this.runningInApp) ? window.innerWidth : this.m_widget.div().offsetWidth;
        var widgetHeight = (this.runningInApp) ? window.innerHeight : this.m_widget.div().offsetHeight;

        if (this.badgeImage)
        {
            var badgeImageEl = $(this.p_badgeImgId());
            var badgeSize = new IWSize(this.badgeImageWidth, this.badgeImageHeight);
            if ((badgeSize.width > widgetWidth) || (badgeSize.height > widgetHeight))
            {
                var widgetSize = new IWSize(widgetWidth, widgetHeight);
                badgeSize = badgeSize.scaleToFit(widgetSize);
            }
            badgeImageEl.width = badgeSize.width;
            badgeImageEl.height = badgeSize.height;
        }

        var overlayNativeWidth = 700;
        var overlayNativeHeight = 286;

        var overlayWidth = Math.max(widgetWidth, overlayNativeWidth);
        var overlayHeight = overlayNativeHeight;
        var overlayTop = Math.min(((widgetHeight / 2) - overlayNativeHeight), 0);
        var overlayLeft = Math.min(((widgetWidth / 2) - (overlayNativeWidth / 2)), 0);

        var overlayImage = $(this.p_overlayImgId());
        overlayImage.width = overlayWidth;
        overlayImage.height = overlayHeight;
        overlayImage.setStyle({left: px(overlayLeft), top: px(overlayTop) });

        var statusMessageBlock = $(this.p_statusMessageBlockId());
        if (statusMessageBlock)
        {
            var leftValue = px(Math.max(((widgetWidth - statusMessageBlock.offsetWidth) / 2), 0));
            var positionStyles = {left: leftValue};
            if (this.statusMessageVerticallyCentered)
            {
                var topValue = px(Math.max(((widgetHeight - statusMessageBlock.offsetHeight) / 2), 0));
                positionStyles.top = topValue;
            }
            statusMessageBlock.setStyle(positionStyles);
        }

        if (this.footerView)
        {
            this.footerView.resize();
        }
    },

    doneFadingIn: function()
    {
        this.m_widget.setPreferenceForKey(true, "x-viewDoneFadingIn", false);
    },

    p_badgeImgId: function()
    {
        return this.m_widget.getInstanceId(this.m_divId + "-badge");
    },

    p_overlayImgId: function()
    {
        return this.m_widget.getInstanceId(this.m_divId + "-overlay");
    },

    p_statusMessageBlockId: function()
    {
        return this.m_widget.getInstanceId(this.m_divId + "-messageBlock");
    }
});

