//
//  iWeb - WidgetInspectability.js
//  Copyright 2006-2008 Apple Inc. All rights reserved.
//

/* in-app only */

Widget.addMethods({ 
    applySelectedStateToStyleRule: function(rule)
    {
        unapplySelectedStateToStyleRule(rule);
        rule.iWebSavedBackgroundColor = rule.style.getPropertyValue("background-color");
        rule.iWebSavedTextColor = rule.style.getPropertyValue("color");
        rule.style.backgroundColor = this.preferenceForKey("selectedTextBackgroundColor");
        rule.style.color = this.preferenceForKey("selectedTextColor");
    },
 
    unselectField: function()
    {
        if (this.selectedStyleRule !== undefined)
        {
            unapplySelectedStateToStyleRule(this.selectedStyleRule);
            delete this.selectedStyleRule;
        }
        this.setPreferenceForKey("", "x-selected-style-class", false);
    },
 
    selectFieldType: function(styleClass)
    {
        if (this.enableSubSelection)
        {
            iWLog("selected field with class " + styleClass);
            this.unselectField();
    
            var rules = findStyleRulesInDocumentForFieldType(styleClass);

            if (rules.length >= 1)
            {
                var rule = rules[0];
                this.applySelectedStateToStyleRule(rule);
                this.selectedStyleRule = rule;
                this.setPreferenceForKey(styleClass, "x-selected-style-class", false);
            }
        }
    },

    onunload: function()
    {
        // todo: this doesn't seem to be called, but it would be quite convenient.
    },

    didBeginEditing: function()
    {
        this.enableSubSelection = true;
        this.unselectField();
        disableLinks(document.body);
        this.addSelectionClickHandlers();
    },

    didEndEditing: function()
    {
        this.enableSubSelection = false;
        this.unselectField();
        enableLinks(document.body);
        this.removeSelectionClickHandlers();
    },

    elementsNeedingClickHandlers: function()
    {
        // Gather together all of the elements that need click handlers.
        var items = $A();
    
        // First, all of the indexed elements prefixed with this.selectableBlockRootName
        var index = 0;
        while(true)
        {
            var itemNode = this.getElementById(this.selectableBlockRootName, index++);
            if (itemNode == null)
            {
                break;
            }
            items.push(itemNode);
        }
    
        // Second, all of the special-case elements listed in this.selectableBlockItems
        $A(this.selectableBlockItems).each(function(item) {
            items.push(this.getElementById(item));
        }.bind(this));
    
        return items;
    },

    itemClickSelectionHandler: function(event) {
        this.unselectField();
    },

    fieldClickSelectionHandler: function(className, event) {
        this.selectFieldType(className);
        event.stop();
    },

    // Go through the current DOM and add selection click handlers to fields. Do this after rendering to the DOM, and
    // when the state of the selection mode changes.
    //
    // To make a field selectable do the following:
    // -- Add an entry to the fields table here with it's element id suffix and SFWP paragraph style.
    // -- Make sure a paragraph style exists in the blog page's stylesheet
    //
    addSelectionClickHandlers: function()
    {
        if(this.selectableClasses)
        {
            // Gather together all of the elements that need click handlers.
            var items = this.elementsNeedingClickHandlers();
        
            // Iterate through the collection we've built, and add the click handlers
            items.each(function(itemNode) {        
                itemNode.observe('click', this.itemClickSelectionHandler.bind(this));
        
                $A(this.selectableClasses).each(function(className)
                {
                    itemNode.select("."+className).each(function(element) {
                        element.observe('click', this.fieldClickSelectionHandler.bind(this, className));
                    }.bind(this));
                }.bind(this));
            }.bind(this));
        }
    },

    removeSelectionClickHandlers: function()
    {
        if(this.selectableClasses)
        {
            // Gather together all of the elements that use click handlers.
            var items = this.elementsNeedingClickHandlers();

            // Iterate through the collection we've built, and remove the click handlers
            items.each(function(itemNode) {
                itemNode.stopObserving('click');
        
                $A(this.selectableClasses).each(function(className)
                {
                    itemNode.select("."+className).each(function(element) {
                        element.stopObserving('click');
                    }.bind(this));
                }.bind(this));
            }.bind(this));
        }
    },

    getStyleElement: function(key)
    {
        // create a style element in the document's head, if we haven't already made one
        if (!this.styleElement)
        {
            var head = document.getElementsByTagName("head")[0];
            if( head )
            {
                var newElement = document.createElement("style");
                newElement.type = "text/css";
                head.appendChild(newElement);
                this.styleElement = newElement;
            }
        }
        
        return this.styleElement;        
    },

    // todo(brentm): seems odd that there's an IE case in a private method...

    // Called by iWeb when the stylesheet changes.
    //
    setDocumentInternalCSS: function(css)
    {
        var styleElement = this.getStyleElement();
        if (styleElement)
        {
            if (!window.windowsInternetExplorer)
            {
                var node = document.createTextNode(css);
                if( node )
                {
                    while (styleElement.hasChildNodes())
                    {
                        styleElement.removeChild(styleElement.firstChild);
                    }
                    styleElement.appendChild(node);
                }
            }
            else
            {
                styleElement.styleSheet.cssText = css;                
            }
        }
    }

});

//
// Utility Methods
//

function findStyleRulesInSheetForFieldType(sheet, type)
{
    var result = [];
    for (var ruleIndex = 0; ruleIndex < sheet.rules.length; ++ruleIndex)
    {
        var rule = sheet.rules[ruleIndex];
        if (rule.type == 1)
        {
            if (rule.selectorText.endsWith("." + type))
            {
                result.push(rule);
            }
        }
        else if (rule.type == 3)
        {
            var rules = findStyleRulesInSheetForFieldType(rule.styleSheet, type);
            result = result.concat(rules);
        }
    }
    return result;
}
 
function findStyleRulesInDocumentForFieldType(type)
{
    var result = [];
    
    for (var sheetIndex = 0; sheetIndex < document.styleSheets.length; ++sheetIndex)
    {
        var rules = findStyleRulesInSheetForFieldType(document.styleSheets.item(sheetIndex), type);
        result = result.concat(rules);
    }
    return result;
}
 
function unapplySelectedStateToStyleRule(rule)
{
    if (rule.iWebSavedBackgroundColor !== undefined)
    {
        if (rule.iWebSavedBackgroundColor === null)
        {
            rule.style.removeProperty('background-color');
        }
        else
        {
            rule.style.backgroundColor = rule.iWebSavedBackgroundColor;
        }
        delete rule.iWebSavedBackgroundColor;
    }
    
    if (rule.iWebSavedTextColor !== undefined)
    {
        if (rule.iWebSavedTextColor === null)
        {
            rule.style.removeProperty('color');
        }
        else
        {
            rule.style.color = rule.iWebSavedTextColor;
        }
        delete rule.iWebSavedTextColor;
    }
}

function enableLinks(node)
{
    var links = $(node).select('a');
    for (var i = 0; i < links.length; ++i)
    {
        var link = links[i];
        var href = link.getAttribute("x-href");
        if (href != undefined)
        {
            link.setAttribute("href", href);
            link.removeAttribute("x-href");
        }
    }
}

function disableLinks(node)
{
    var links = $(node).select('a');
    for (var i = 0; i < links.length; ++i)
    {
        var link = links[i];
        var href = link.href;
        if( href != undefined )
        {
            link.setAttribute("x-href", href);
            link.removeAttribute("href");
        }
    }
}
