//
// iWeb - BlogSummary.js
// Copyright 2006-2008 Apple Inc.
// All rights reserved.
//

// Responsibility: Brent Marykuca
// Reviewers: brentm

var BlogSummaryWidget = Class.create(JSONFeedRendererWidget, {

    widgetIdentifier: "com-apple-iweb-widget-blogSummary",

    initialize: function($super, instanceID, widgetPath, sharedPath, sitePath, preferences, runningInApp)
    {
        if (instanceID)
        {
            $super(instanceID, widgetPath, sharedPath, sitePath, preferences, runningInApp);

            this.contentID  = "summary-content";
            this.itemTemplateID = "item-template";
            this.itemID = "item";
            this.separatorTemplateID = "separator-template";
            this.selectState = null; // start out with nothing selected
            this.sfrShadow = null;
            this.disableShadows = isEarlyWebKitVersion;  // Shadows are too slow with older webkit versions
            this.styleThreeMaximumWidth = 0.65;
            this.debugBorders = (location.href.indexOf("summaryBorders") != -1);
            this.debugCompiledHtml = (location.href.indexOf("summaryCompiledHtml") != -1);
            this.debugFinalHtml = (location.href.indexOf("summaryFinalHtml") != -1);
            this.debugTrace = (location.href.indexOf("summaryTrace") != -1);
        
            if (this.debugTrace) trace = print;

            if (this.privateInitialize)
            {
                this.privateInitialize();
            }
       
            document.blogSummaryWidget = this;
        
            this.initializingPreferences = true;
            this.initializeDefaultPreferences({
                "htmlTemplate": "",
                "excerpt-length": "100",
                "extraSpace": 0,
                "headerOnTop": false,
                "imageSize": "25",
                "imageSizeOverride": "",
                "imagePosition": "Left",
                "imageOutside": false,
                "imageVisibility": true,
                "rss-for-archive": false,
                "photoProportions": "Landscape"
            });
            delete this.initializingPreferences;

            this.dotMacAccount = this.preferenceForKey("dotMacAccount") || "";
        
            this.updateTemplate();
            this.applyUpgrades();
            this.fetchAndRenderRSS();
        }
    },

    // When size changes, we rerender the items, but don't need to reload the RSS
    sizeDidChange: function()
    {
        this.invalidateFeedItems("sizeDidChange");
    },

    onload: function($super)
    {
        $super();

        this.changedPreferenceForKey("htmlTemplate");
    
        this.changedPreferenceForKey("sfr-stroke");
        this.changedPreferenceForKey("sfr-reflection");
        this.changedPreferenceForKey("sfr-shadow"); // do this last because it will force an immediate render
    },

    changedPreferenceForKey: function($super, key)
    {
        try
        {
            if (this.initializingPreferences)
            {
                return;
            }

            if (this.privateChangedPreferenceForKey)
            {
                this.privateChangedPreferenceForKey(key);
            }

            if (key == "htmlTemplate")
            {
                this.updateTemplate();
            }
    
            if (key == "rss-for-archive")
            {
                this.fetchAndRenderRSS();
            }
    
            if (key == "photoProportions")
            {
                this.invalidateFeedItems(key);
            }
        }
        catch (e)
        {
            debugPrintException(e);
        }

        // Call superclass' changedPreferenceForKey
        $super(key);
    },

    applyUpgrades: function()
    {
        var upgrades = this.preferenceForKey("upgrades");
        if (upgrades === undefined)
        {
            upgrades = {};
        }
    
        if ((!upgrades["styleThreeWidthScale"]))
        {
            // After content freeze, we changed the maximum value of the image placement slider to 65% of the
            // width of the widget. (See <rdar://problem/5095842>) We upgrade the value from the template here so
            // they maintain the right size visually.
            //
            // New Value =  Old Value / 0.65
        
            if (this.imagePlacementStyle() == 3)
            {
                var oldImageSize = this.preferenceForKey("imageSize");
                var newImageSize = Math.floor(oldImageSize / this.styleThreeMaximumWidth);
                this.setPreferenceForKey(newImageSize, "imageSize", false);
                trace("Upgrading image size from %s to %s", oldImageSize, newImageSize);
            }
            upgrades["styleThreeWidthScale"] = true;
            this.setPreferenceForKey(upgrades, "upgrades", false);
        }
    },

    updateTemplate: function()
    {
        var html = this.preferenceForKey("htmlTemplate");
        html = html.replace(/<div class=["']clear:[ ]*both;?['"] ?\/>/g, "<div style='clear:both'></div>"); // error in goldenrod: 'class' instead of style
        html = html.replace(/<div style=["']clear:[ ]*both;?['"] ?\/>/g, "<div style='clear:both'></div>");

        if (html == "")
        {
            var markup;
            if (this.preferenceForKey("rss-for-archive"))
                markup = BlogSummaryShared.defaultArchiveMarkup;
            else
                markup = BlogSummaryShared.defaultSummaryMarkup;
            
            trace("** brand-new widget html being compiled from source markup");
            if (markup)
            {
                html = BlogSummaryShared.compileMarkup(markup);
            }
        }
        if (html)
        {
            // <rdar://problem/5156857> Blog/Podcast Summary/Archive: "Read More..." is not localizable
            // Apply localization to "Read more..." or "More..." text if it's not already there.
            if (html.indexOf("<span class='bl-localized'>") == -1)
            {
                if (html.indexOf("Read more...") != -1)
                    html = html.replace(/Read more\.\.\./g, "<span class='bl-localized'>Read more...</span>");
                else if (html.indexOf("More...") != -1)
                    html = html.replace(/More\.\.\./g, "<span class='bl-localized'>More...</span>");
            }

            var contentDiv = this.getElementById(this.contentID);
            if (contentDiv)
            {
                if (this.debugCompiledHtml) print("\n\n%s\n\n", html);
                html = html.replace(/\$WIDGET_ID/g, this.instanceID);
                contentDiv.innerHTML = html;

                this.resetRenderingState = true;
                this.invalidateFeedItems("updateTemplate");

                // We don't want any of our content to be PNG fixed by the global background png fix that runs at the
                // end of page load, so mark every DIV to be skipped. We do a separate fix after we finish rendering.
                //
                optOutOfCSSBackgroundPNGFix(contentDiv);
            }
        }
    },

    commentCountText: function(count, enabled)
    {
        if (count == 0)
        {
            if (enabled)
            {
                return this.localizedString("No Comments");
            }
            else
            {
                return "";
            }
        }
        else if (count == 1)
        {
            return this.localizedString("1 Comment");
        }
        else
        {
            return String.stringWithFormat(this.localizedString("%s Comments"), count);
        }
    },

    replaceLinkTargets: function(node, replaceArray)
    {
        var byteToPercentEscapeValue = function(ch)
        {
            ch = ch & 0xFF;
            var hiChar = "0123456789ABCDEF".charAt(ch / 16);
            var loChar = "0123456789ABCDEF".charAt(ch % 16);
            return "%" + hiChar + loChar;
        };

        var sanitizeUrlStringForIE6 = function(s)
        {
            var result = "";
            for (var i = 0; i < s.length; ++i)
            {
                var ch = s.charCodeAt(i);
                if (ch < 128)
                    result += s.charAt(i);
                else
                    result += byteToPercentEscapeValue(ch);
            }
            return result;
        };

        var links = $(node).select('a');
        for (var j = 0; j < links.length; ++j)
        {
            var linkTarget = links[j].getAttribute("href");
            if (linkTarget && linkTarget.length > 0)
            {
                for (var i = 0; i < replaceArray.length; ++i)
                {
                    linkTarget = linkTarget.replace(replaceArray[i][0], replaceArray[i][1]);
                }
                if (linkTarget)
                {
                    if (windowsInternetExplorer && effectiveBrowserVersion < 7)
                    {
                        // <rdar://problem/5425706> Japanese: Blog/Podcast summary entry image and title links to a missing iWeb page in IE XP
                        // Convert non-ASCII bytes to percent-escapes in the link target. We use a custom function to do this rather than 
                        // encodeURI() because encodeURI() treats extended ASCII characters as if they were Latin-1 and converts them to UTF-8
                        // encoded byte sequences. At least, that's what it does in IE6.
                        linkTarget = sanitizeUrlStringForIE6(linkTarget);
                    }
                    links[j].setAttribute("href", linkTarget);
                }
            }
        }
    },

    renderCommentCount: function(itemNode, index, enabled, count)
    {
        var commentCountText = this.commentCountText(count, enabled);
    
        var spans = itemNode.select(".bl-value-comment-count");
        $A(spans).each(function(node)
        {
            node.update(commentCountText);
        });
    
        var commentFieldNode = this.getElementById("comment-field", index);
        if (commentFieldNode)
        {
            commentFieldNode.style.display = (commentCountText.length === 0) ? "none" : "";
        }
    },

    applyLocalization: function(parent)
    {
        var localizedSpans = $(parent).select(".bl-localized");
        localizedSpans.each(function(span)
        {
            var key = getTextFromNode(span);
            span.update(this.localizedString(key));
        }.bind(this));
    },

    // Return a value in the range 1..5 describing the image placement style implied by the current
    // preferences. These styles are usually referred to by their position on the HUD menu.
    //
    imagePlacementStyle: function()
    {
        var result;
        var headerOnTop        = this.preferenceForKey("headerOnTop");
        var imageOutside       = this.preferenceForKey("imageOutside");
        var imagePosition      = this.preferenceForKey("imagePosition");
        var imageSizeOverride  = this.preferenceForKey("imageSizeOverride");
    
        var settings = [imagePosition, imageOutside, headerOnTop, imageSizeOverride != ''];
        if (settings.isEqual(['Left', false, true, false]))
            result = 1;
        else if (settings.isEqual(['Left', false, false, false]))
            result = 2;
        else if (settings.isEqual(['Left', true, false, false]))
            result = 3;
        else if (settings.isEqual(['Right', false, false, false]))
            result = 4;
        else if (settings.isEqual(['Left', false, true, true]))
            result = 5;

        return result;
    },

    renderFeedItems: function()
    {
        trace('renderFeedItems(%s)', arguments[0]);
    
        // If a deferred render is pending, then cancel it because we're about to render now.
        //
        if (this.pendingRender)
        {
            clearTimeout(this.pendingRender);
            this.pendingRender = null;
        }
    
        // If onload hasn't been called yet, then defer rendering awhile. We've noticed problems in Safari where
        // some divs' offsetWidth/height values aren't valid until after onload.
        //
        if (this.onloadReceived == false)
        {
            this.invalidateFeedItems();
            return;
        }
    
        if (this.blogFeed.itemCount() === 0)
        {
            trace(' exit: no items');
            return;
        }

        if (this.resetRenderingState)
        {
            this.lastRerenderImageSettings =  [];
            this.lastRedoLayoutSettings = [];
            this.resetRenderingState = false;
        }
    
        var itemTemplateNode = this.getElementById(this.itemTemplateID);
        var separatorTemplateNode = this.getElementById(this.separatorTemplateID);
    
        if (itemTemplateNode === null) 
        {
            trace(' exit: no template');
            return;
        }

        var parentNode = itemTemplateNode.parentNode;

        var shouldDisableLinks = this.enableSubSelection;
        var excerptLength      = this.preferenceForKey("excerpt-length");  // can be -100 to represent "all content"
        var imageSize          = this.preferenceForKey("imageSize"); // percentage of widget width in range 10..100
        var imageSizeOverride  = this.preferenceForKey("imageSizeOverride");
        var imagePosition      = this.preferenceForKey("imagePosition");
        var showPhotosOption   = this.preferenceForKey("imageVisibility");
        var extraSpace         = this.preferenceForKey("extraSpace");
        var imageOutside       = this.preferenceForKey("imageOutside");
        var headerOnTop        = this.preferenceForKey("headerOnTop");
        var photoProportions   = this.preferenceForKey("photoProportions");
        var isArchive          = this.preferenceForKey("rss-for-archive");
    
        var sfrShadowText      = this.preferenceForKey("sfr-shadow");
        var sfrReflectionText  = this.preferenceForKey("sfr-reflection");
        var sfrStrokeText      = this.preferenceForKey("sfr-stroke");

        var showImages         = showPhotosOption && (excerptLength !== 0);
    
        var imagePlacementStyle = this.imagePlacementStyle();
    
        if (this.preferences && this.preferences.postNotification)
            this.preferences.postNotification("BLWidgetIsSafeToDrawNotification", 0);
                                     
        if (this.runningInApp && ((this.lastImagePlacementStyle !== undefined) && (this.lastImagePlacementStyle != imagePlacementStyle)))
        {
            var tempLastImagePlacementStyle = this.lastImagePlacementStyle;
            this.lastImagePlacementStyle = imagePlacementStyle;
            if (imagePlacementStyle == 3)
            {
                // going TO placement style 3, scale the image size up
                imageSize = Math.min(Math.ceil(imageSize / this.styleThreeMaximumWidth), 100);
                this.setPreferenceForKey(imageSize, "imageSize", false); /*forces re-render, so return right away*/
                return;
            }
            else if (tempLastImagePlacementStyle == 3)
            {
                // going FROM placement style 3, scale the size down
                imageSize = Math.max(10, Math.ceil(imageSize * this.styleThreeMaximumWidth));
                this.setPreferenceForKey(imageSize, "imageSize", false); /* see above */
                return;
            }
        }
        this.lastImagePlacementStyle = imagePlacementStyle;
    
        // Style 3 renders just like Style 2 if photos are turned off
        if ((imagePlacementStyle == 3) && !showImages)
        {
            imageOutside = false;
            imagePlacementStyle = 2;
        }
        // Scale the image size for placement 3. Maximum value is 65% of width. 
        if (this.imagePlacementStyle() == 3)
        {
            var oldImageSize = imageSize;
            imageSize = Math.min(Math.max(10, Math.ceil(imageSize * this.styleThreeMaximumWidth)), 100);
            trace("imageSize is %s but using %s because this is style 3", oldImageSize, imageSize);
        }
    
        if (imageSizeOverride !== "")
        {
            imageSize = Number(imageSizeOverride);
        }

    
        // Some themes have padding that they want to apply to summary items. Such padding should effectively reduce
        // what "100%" means when an image size is being calculated.
    
        var paddingLeft = 0;
        var paddingRight = 0;
    
        var contentDiv = this.getElementById(this.contentID);
        if (this.debugBorders) contentDiv.style.border = "1px solid orange";
    
        var node = this.getElementById("image");
        if (node)
        {
            while ((node !== null) && (node !== contentDiv))
            {
                if (node.style)
                {
                    paddingLeft += parseFloat(node.style.paddingLeft || 0);
                    paddingRight += parseFloat(node.style.paddingRight || 0);
                }
                node = node.parentNode;
            }
        }
    
        var imageWidth  = imageSize / 100.0 * ($(parentNode).getWidth() - (paddingLeft + paddingRight));

        if (imageSizeOverride !== "" && this.sfrStroke)
        {
            var strokeExtra = this.sfrStroke.strokeExtra();
            imageWidth -= (strokeExtra.left + strokeExtra.right);
        }
        if (this.preferences && this.preferences.postNotification)
        {
            this.preferences.postNotification("BLBlogSummaryWidgetThumbnailWidthNotification", imageWidth);
        }

        var rerenderImageSettings = [
            extraSpace, showImages, imageWidth, imagePosition, imageOutside, headerOnTop, 
            sfrShadowText, sfrReflectionText, sfrStrokeText, photoProportions
        ];

        // Note: imagePosition, imageOutside and headerOnTop require image to rerender only in order to work around
        // a bug(?) in webkit layout where a canvas is not repositioned when the dom changes. To see this happen, 
        // remove those those three keys from rerenderImageSettings and try playing with the layout controls
        // on the HUD.
        var redoLayoutSettings = [imagePosition, extraSpace, imageOutside, imageWidth, headerOnTop, sfrStrokeText, 
            showImages];
    
        var rerenderImage = (this.lastRerenderImageSettings === undefined || 
                             (rerenderImageSettings.isEqual(this.lastRerenderImageSettings) == false));
        var redoLayout = (this.lastRedoLayoutSettings === undefined ||
                               (redoLayoutSettings.isEqual(this.lastRedoLayoutSettings) == false));

        this.lastRerenderImageSettings = rerenderImageSettings;
        this.lastRedoLayoutSettings = redoLayoutSettings;
    
        var maxItems = this.blogFeed.maximumItemsToDisplay();
    
        for (var i = 0; i < maxItems; ++i)
        {
            var item = this.blogFeed.itemAtIndex(i);
            var itemNode = this.getElementById(this.itemID, i);
            if (itemNode === null)
            {
                // clone the template node, adjust its ids and insert into the DOM
                itemNode = itemTemplateNode.cloneNode(true/*deep*/);
                itemNode.id = this.getInstanceId(this.itemID);
                adjustNodeIds(itemNode, i);

                // New nodes are inserted at the end of the parent before the summary template node

                // Placement style 3 floats the item right, so be sure to clear floats before adding a new item
                // otherwise they can stack up in multiple columns. <radar:5155185>. We do this in all cases
                // because users might switch styles to placement 3 after the node is created.
                var divClearBoth = document.createElement("div");
                divClearBoth.style.clear = "both";
                parentNode.insertBefore(divClearBoth, itemTemplateNode);

                parentNode.insertBefore(itemNode, itemTemplateNode);
            }
        
            // Note: Item node is in the DOM now.
        
            if (shouldDisableLinks)
            {
                disableLinks(itemNode);
            }
        
            // Format the date as a string
            if (!item.dateString)
            {
                if (item.date)
                {
                    item.dateString = item.date.stringWithICUDateFormat(this.blogFeed.dateFormat(), this /*localizer*/);
                }
            }
        
            substituteSpans(itemNode,
            {
                "bl-value-title":   ["html", item.title || ""],
                "bl-value-date":    ["text", item.dateString],
                "bl-value-excerpt": ["html", this.summaryExcerpt(item.description || "", excerptLength)]
            });
        
            if (this.runningInApp)
            {
                this.renderCommentCount(itemNode, i, item.commentingEnabled, item.commentCount);

                // Replace link attributes
                this.replaceLinkTargets(itemNode, [[/\$link\$/g, item.relativeURL.toURLString()], 
                                                   [/\$comment-link\$/g, item.relativeCommentURL.toURLString()]]);
            }
            else
            {
                var linkReplacer = function(item, itemNode)
                {
                    // Replace link attributes
                    var itemCommentLink = item.relativeURL.toURLString() + "#comment_layer";
                    this.replaceLinkTargets(itemNode, [[/\$link\$/g, item.relativeURL.toURLString()], [/\$comment-link\$/g, itemCommentLink]]);
                }.bind(this, item, itemNode);
            
                // Try to fetch comments before doing link replacement if this was published to Dot Mac.
                if (this.dotMacAccount != "")
                {
                    this.fetchCommentCountInfoForItem(item, i, itemNode, linkReplacer);
                }
                else
                {
                    linkReplacer();
                }
            }

            var isEquivalentToEmpty = function(s)
            {
                return (s === undefined) || (s === null) || (s === "");
            }

            var entryHasImage = (! isEquivalentToEmpty(item.imageUrlString));
            var badgeType = item.badgeType;
        
            var imgDiv = $(this.getElementById("image", i));
            var imgGroupDiv = $(this.getElementById("image-group", i));
        
            var headerNode = $(this.getElementById("header", i));

            if (headerNode)
            {
                if (headerOnTop)
                {
                    // put the image div after the headerNode
                    imgDiv.parentNode.insertBefore(headerNode, imgDiv);
                }
                else
                {
                    // put the image div before the header node
                    imgDiv.parentNode.insertBefore(headerNode, imgDiv.nextSibling);
                }
            }

            if (imgDiv)
            {
                if (showImages)
                {
                    if (rerenderImage)
                    {
                        this.rerenderImage(imgGroupDiv, imgDiv, item.imageUrlString, entryHasImage, 
                            photoProportions, imageWidth, this.addBadge.bind(this, badgeType, imgDiv));
                    }
                }
            
                if (redoLayout)
                {
                    $(imgDiv).setStyle({ cssFloat: imagePosition });
                    $(itemNode).setStyle({ cssFloat: "none" });
                
                    var strokeWidth = 0;
                    var strokeHeight = 0;
                    if (this.sfrStroke)
                    {
                        // get the stroke width/height even if we don't have an image. this affects the layout in style 3
                        var strokeExtra = null;
                        if (this.sfrStroke.strokeExtra)
                        {
                            strokeExtra = this.sfrStroke.strokeExtra(); 
                        }
                        if (strokeExtra)
                        {
                            strokeWidth = (strokeExtra.left + strokeExtra.right);
                            strokeHeight = (strokeExtra.top + strokeExtra.bottom);
                        }
                    }

                    if (imageOutside)
                    {
                        if (imagePosition == "Left")
                        {
                            // This is it! The infamous "Style #3".
                            var leftInset = (imageWidth + extraSpace + strokeWidth);

                            if ((windowsInternetExplorer && effectiveBrowserVersion < 7) || isFirefox)
                                leftInset += paddingLeft;

                            var marginLeft = -1 * leftInset;

                            $(imgDiv).setStyle({
                                marginLeft: px(marginLeft),
                                marginRight: 0,
                                paddingBottom: 0,
                                paddingLeft: 0,
                                paddingRight: 0
                            });
                            if (this.debugBorders) imgDiv.style.border = "1px solid blue";
                        
                            if (windowsInternetExplorer && effectiveBrowserVersion < 7)
                                itemNode.style.paddingLeft  = 0;
                            else
                                itemNode.style.paddingLeft  = px(imageWidth + extraSpace + strokeWidth + paddingLeft);
                            
                            itemNode.style.paddingRight = 0;
                            if (this.debugBorders) itemNode.style.border = "1px solid red";
                        
                            // For <rdar://problem/5128622>: Set the width in style 3 to avoid right aligned items in the grid.
                            // We also clear the width in all other styles.
                            itemNode.style.width        = px(depx(this.div().style.width) - leftInset - paddingLeft - paddingRight);
                        
                            // ImageOutside/Left doesn't work at all on IE7 unless the item node is floated right.
                            // (Otherwise IE7 completely ignores the negative margin on the imgDiv.)
                            $(itemNode).setStyle({ cssFloat: "right" });

                        }
                        else if (imagePosition == "Right")
                        {
                            $(imgDiv).setStyle({
                                marginLeft: 0,
                                marginRight: px(-1 * (imageWidth + extraSpace)),
                                paddingBottom: 0,
                                paddingLeft: 0,
                                paddingRight: 0
                            });
                        
                            $(itemNode).setStyle({
                                paddingLeft: 0,
                                paddingRight: px(imageWidth + extraSpace),
                                width: ""
                            });
                        }
                    }
                    else
                    {
                        if (imagePosition == "Left")
                        {
                            $(imgDiv).setStyle({
                                marginLeft: 0,
                                marginRight: 0,
                                paddingBottom: px(extraSpace + strokeHeight),
                                paddingLeft: 0,
                                paddingRight: px(extraSpace + strokeWidth)
                            });
                        
                            $(itemNode).setStyle({
                                paddingLeft: 0,
                                paddingRight: 0,
                                width: ""
                            })
                        }
                        else if (imagePosition == "Right")
                        {
                            $(imgDiv).setStyle({
                                marginLeft: 0,
                                marginRight: 0,
                                paddingBottom: px(extraSpace + strokeHeight),
                                paddingLeft: px(extraSpace),
                                paddingRight: px(strokeWidth)
                            });
                        
                            $(itemNode).setStyle({
                                paddingLeft: 0,
                                paddingRight: 0,
                                width: ""
                            });
                        }
                    }
                }
            
                if (showImages && entryHasImage)
                {
                    if (imgDiv.style.display != "")
                    {
                        imgDiv.style.display = "";
                    }
                }
                else
                {
                    if (imgDiv.style.display != "none")
                        imgDiv.style.display = "none";

                    itemNode.style.paddingLeft  = 0;
                    itemNode.style.paddingRight = 0;
                }
            }
        
            // End of summary image handling
        
            // Separators
            //
            var newSep = null;
            if (separatorTemplateNode && (i < this.blogFeed.itemCount() - 1))
            {
                newSep = this.getElementById(this.separatorTemplateID, i);
                if (newSep == null)
                {
                    newSep = separatorTemplateNode.cloneNode(true); // true=deep
                    adjustNodeIds(newSep, i);
                    parentNode.insertBefore(newSep, itemTemplateNode);
                }
            }

            this.applyLocalization(itemNode);
        
            // Finally, clear 'display: none' style to show the completed summary entry
        
            itemNode.style.display = "";
            if (newSep)
            {
                newSep.style.display = "";
            }
        }
        // End of item loop
    
        // Remove nodes past the end in case the item list got smaller.
        var index = maxItems;
        while (true)
        {
            if (index > 0)
            {
                var sep = this.getElementById(this.separatorTemplateID, index - 1);
                if (sep)
                {
                    sep.parentNode.removeChild(sep);
                }
            }

            var node = this.getElementById(this.itemID, index);
            if (node === null)
            {
                break;
            }
            node.parentNode.removeChild(node);
            index++;
        }
    
        // <rdar://problem/5169216> BREAK: Main Event: Blog/Podcast Summary: visual problems with the stars in the widget
        // Part of the problem in this radar was that the background row of stars didn't extend to the bottom of the widget
        // because the entire item is floated right (in style 3) and there is no content to expand the box. To make this work,
        // we add a span at the end of the content div (after the template div) and show and hide it depending on the current
        // placement style.
        var pusherSpan = $("pusher");
        if (pusherSpan == null)
        {
            pusherSpan = document.createElement("span");
            pusherSpan.innerHTML = "&nbsp;";
            pusherSpan.id = "pusher";
            itemTemplateNode.parentNode.insertBefore(pusherSpan, null);
        }
        pusherSpan.style.display = (imagePlacementStyle == 3) ? "" : "none";
    
        if (windowsInternetExplorer)
        {
            fixAllIEPNGs(transparentGifURL());
            setTimeout(fixupIEPNGBGsInTree.bind(null, contentDiv, true/*force*/), 1);
            if (effectiveBrowserVersion >= 8)
            {
                joltLater(this.div());
            }
        }

        if (this.privateSummaryDidRender)
        {
            this.privateSummaryDidRender();
        }
        if (this.debugFinalHtml)
        {
            // Wait 5 seconds (for any in-progress rendering to finish) and then dump the HTML.
            setTimeout(function() 
            {
                print(contentDiv.outerHTML);
            }, 5000);
            this.debugFinalHtml = false;
        }
        trace(" exit: done", this.blogFeed.itemCount(), "items");
    },

    fetchAndRenderRSS: function()
    {
        var isArchiveWidget = this.preferenceForKey("rss-for-archive");
        
        this.blogFeed = new BlogFeed(BlogRootURLString(location.href), isArchiveWidget, function()
        {
            this.invalidateFeedItems("renderSummaryFromRSS");

        }.bind(this));
    },

    // Scrape the comment info from .Mac and fill in the comment count field. We only do this online;
    // in-app we get comment count information from the RSS feed.
    //
    fetchCommentCountInfoForItem: function(item, index, itemNode, postRenderCallback)
    {
        var commentSummaryURL = item.absoluteURL.toURLString() + "?wsc=summary.js&ts=" + new Date().getTime();

        var renderItemCommentSummary = function(request)
        {
            // We XHR the old comment summary script fragments and match the two fields that
            // we need.
            if (request.responseText)
            {
                var r = request.responseText.match(/.*= ((true)|(false));.*\n.*= (\d+)/);
                if (r)
                {
                    var enabled = (r[1] == "true");
                    var count = Number(r[4]);
                    this.renderCommentCount(itemNode, index, enabled, count);
                }
            }
            postRenderCallback();
        }.bind(this);

        new Ajax.Request(commentSummaryURL, {
            method: 'get',
            onComplete: renderItemCommentSummary
        });
    },

    addBadge: function(badgeType, imageDiv)
    {
        if (badgeType == "movie" || badgeType == "audio")
        {
            var cropDiv = imageDiv.down('.crop');

            // Create a div that looks like a translucent black bar across the bottom of the image
            var kBadgeHeight = 16.0;
            var blackBarElem = new Element('div', {className: 'badge-fill'});
            blackBarElem.setStyle(
                { backgroundColor: 'black',
                  opacity: 0.75,
                  position: 'absolute', 
                  left: 0, 
                  bottom: 0,
                  width: px(cropDiv.offsetWidth),
                  height: px(kBadgeHeight) });
            cropDiv.appendChild(blackBarElem);

            // Place the badge at the left edge of the black bar
            var badgeImageFilename = ( ( badgeType == "movie" ) ? "Overlay-Movie.png" : "Overlay-Audio.png" );
            var badgeSrc = this.widgetPath + "/" + badgeImageFilename;
            var badgeElem = new Element('img', {className: 'badge-overlay', src: badgeSrc});
            badgeElem.setStyle({ position: 'absolute', bottom: 0, left: 0 });
            if(windowsInternetExplorer)
            {
                 badgeElem.setStyle(" filter", "progid:DXImageTransform.Microsoft.AlphaImageLoader(src='" + IEConvertURLForPNGFix(badgeSrc) + " ', sizingMethod='scale');" );
                 badgeElem.writeAttribute(badgeSrc, transparentGifURL());
            }

            cropDiv.appendChild(badgeElem);
        }
    }
});
