//
//  iWeb - Stepper.js
//  Copyright (c) 2007-2008 Apple Inc. All rights reserved.
//

var Stepper = Class.create({
    initialize: function(stepper, width, height,
                         imgUp, imgUpClicked, imgUpHeight, 
                         imgDown, imgDownClicked, imgDownHeight,
                         onValueIncremented, onValueDecremented)
    {    
        if (stepper == null)
            return;
    
        /* Read-write properties */
        this.onValueIncremented = onValueIncremented;
        this.onValueDecremented = onValueDecremented;
    
        /* Read-only properties */
        this.enabled = true;
    
        this._init(stepper, width, height, 
                   imgUp, imgUpClicked, imgUpHeight,
                   imgDown, imgDownClicked, imgDownHeight);
    },

    _init: function(stepper, width, height,
                    imgUp, imgUpClicked, imgUpHeight,
                    imgDown, imgDownClicked, imgDownHeight)
    {
        this._imgUpPath = imgUp;
        this._imgUpClickedPath = imgUpClicked;
        this._imgDownPath = imgDown;
        this._imgDownClickedPath = imgDownClicked;
    
        var container = document.createElement("div");
        var style = container.style;
        style.width = px(width);
        style.height = px(height);
        this._container = container;

        stepper.appendChild(container);

    
        // Create the inner elements    
        var element = document.createElement("div");
        var style = element.style;
        style.display = "inline-block";
        style.background = "url(" + this._imgUpPath + ") no-repeat top left";
        style.height = px(imgUpHeight);
        style.width = px(width);
        container.appendChild(element);
        this.upButton = element;
    
        element = document.createElement("div");
        style = element.style;
        style.display = "inline-block";
        style.background = "url(" + this._imgDownPath + ") no-repeat bottom left";
        style.height = px(imgDownHeight);
        style.width = px(width);
        container.appendChild(element);
        style = container.style;
        this.downButton = element;
    
        // preload clicked images
        var img = new Image(width, imgUpHeight);
        img.src = this._imgUpClickedPath;
        img = new Image(width, imgUpHeight);
        img.src = this._imgDownClickedPath;
    
        // For JavaScript event handlers
        var _self = this;
        this._mousedownInUpHandler = function(event) { _self._mousedown(event, true); }
        this._mousedownInDownHandler = function(event) { _self._mousedown(event, false); }
        this._mousemoveHandler = function(event)
        {
            event.stopPropagation();
            event.preventDefault();
        }
        this._mouseoverInUpHandler = function(event) { _self._mouseover(event, true); }
        this._mouseoverInDownHandler = function(event) { _self._mouseover(event, false); }
        this._mouseoutInUpHandler = function(event) { _self._mouseout(event, true); }
        this._mouseoutInDownHandler = function(event) { _self._mouseout(event, false); }
        this._mouseupInUpHandler = function(event) { _self._mouseup(event, true); }
        this._mouseupInDownHandler = function(event) { _self._mouseup(event, false); }
        this._intervalElapsedInUpHandler = function(event) { _self._intervalElapse(event, true); }
        this._intervalElapsedInDownHandler = function(event) { _self._intervalElapse(event, false); }

        this.upButton.addEventListener("mousedown", this._mousedownInUpHandler, true);
        this.downButton.addEventListener("mousedown", this._mousedownInDownHandler, true);
    },

    remove: function()
    {    
        var parent = this._container.parentNode;
        parent.removeChild(this._container);
    },

    setDisabledImages: function(imgUpDisabled, imgDownDisabled)
    {
        this._imgUpDisabledPath = imgUpDisabled;
        this._imgDownDisabledPath = imgDownDisabled;
    },

    setEnabled: function(enabled)
    {
        this.enabled = enabled;
        if (enabled)
        {
            this._container.children[0].style.backgroundImage = "url(" + this._imgUpPath + ")";
            this._container.children[1].style.backgroundImage = "url(" + this._imgDownPath + ")";
        }
        else if (this._imgUpDisabledPath !== undefined)
        {
            this._container.children[0].style.backgroundImage = "url(" + this._imgUpDisabledPath + ")";
            this._container.children[1].style.backgroundImage = "url(" + this._imgDownDisabledPath + ")";
        }
    },


    /*********************
    * Private handlers
    */

    _setUpPressed: function(pressed)
    {
        if (pressed)
        {
            this._container.children[0].style.backgroundImage = "url(" + this._imgUpClickedPath + ")";
        }
        else
        {
            this._container.children[0].style.backgroundImage = "url(" + this._imgUpPath + ")";
        }
    },

    _setDownPressed: function(pressed)
    {
        if (pressed)
        {
            this._container.children[1].style.backgroundImage = "url(" + this._imgDownClickedPath + ")";
        }
        else
        {
            this._container.children[1].style.backgroundImage = "url(" + this._imgDownPath + ")";
        }
    },


    _mousedown: function(event, up)
    {
        // If we're disabled, don't do anything
        if (!this.enabled)
        {
            event.stopPropagation();
            event.preventDefault();
            return;
        }
    
        // Change images to clicked state
        if (up)
        {
            this._setUpPressed(true);
            this._insideUp = true;

            // add temp event listeners
            document.addEventListener("mousemove", this._mousemoveInUpHandler, true);
            document.addEventListener("mouseup", this._mouseupInUpHandler, true);
            this._container.addEventListener("mouseover", this._mouseoverInUpHandler, true);
            this._container.addEventListener("mouseout", this._mouseoutInUpHandler, true);
            this._timerInterval = setInterval(this._intervalElapsedInUpHandler, 500);
        }
        else
        {
            this._setDownPressed(true);
            this._insideDown = true;

            // add temp event listeners
            document.addEventListener("mousemove", this._mousemoveInDownHandler, true);
            document.addEventListener("mouseup", this._mouseupInDownHandler, true);
            this._container.addEventListener("mouseover", this._mouseoverInDownHandler, true);
            this._container.addEventListener("mouseout", this._mouseoutInDownHandler, true);
            this._timerInterval = setInterval(this._intervalElapsedInDownHandler, 500);
        }
    
        event.stopPropagation();
        event.preventDefault();
    },

    _mouseover: function(event, up)
    {
        // Change images to clicked state
        if (up)
        {
            this._setUpPressed(true);
            this._insideUp = true;
        }
        else
        {
            this._setDownPressed(true);
            this._insideDown = true;
        }
    
    
        event.stopPropagation();
        event.preventDefault();        
    },

    _mouseout: function(event, up)
    {
        // Change images to regular state
        if (up)
        {
            this._setUpPressed(false);
            this._insideUp = false;
        }
        else
        {
            this._setDownPressed(false);
            this._insideDown = false;
        }
    
        event.stopPropagation();
        event.preventDefault();    
    },

    _mouseup: function(event, up)
    {
        // Change images to regular state
        this._setUpPressed(false);
        this._setDownPressed(false);
    
        // Remove temp event listeners
        document.removeEventListener("mousemove", this._mousemoveHandler, true);
        if (up)
        {
            document.removeEventListener("mouseup", this._mouseupInUpHandler, true);
            this._container.removeEventListener("mouseover", this._mouseoverInUpHandler, true);
            this._container.removeEventListener("mouseout", this._mouseoutInUpHandler, true);
        }
        else
        {
            document.removeEventListener("mouseup", this._mouseupInDownHandler, true);
            this._container.removeEventListener("mouseover", this._mouseoverInDownHandler, true);
            this._container.removeEventListener("mouseout", this._mouseoutInDownHandler, true);
        }
    
        // Perform callback if we're inside the appropriate stepper button
        try {
            if (up && this._insideUp && this.onValueIncremented != null)
            {
                this.onValueIncremented(event);
            }
            else if (!up && this._insideDown && this.onValueDecremented != null)
            {
                this.onValueDecremented(event);
            }
        } catch(ex) {
            throw ex;
        } finally {
            event.stopPropagation();
            event.preventDefault();
            delete this._insideUp;
            delete this._insideDown;
        }
        clearInterval(this._timerInterval);
    },

    _intervalElapse: function(event, up)
    {
        clearInterval(this._timerInterval);
        if (up && this._insideUp && this.onValueIncremented != null)
        {
            this.onValueIncremented(event);
            this._timerInterval = setInterval(this._intervalElapsedInUpHandler, 100);
        }
        else if (!up && this._insideDown && this.onValueDecremented != null)
        {
            this.onValueDecremented(event);
            this._timerInterval = setInterval(this._intervalElapsedInDownHandler, 100);
        }
    }
});
