//
//  iWeb - TimerWidget.js
//  Copyright (c) 2007-2008 Apple Inc. All rights reserved.
//

/*
Classes hierarchy:

    Widget
        TimerWidget

    TimerView
        TextualTimerView                // Text Timer
        GraphicalTimerView 
            DigitalTimerView            // Digital Clock
            FlipTimerView               // Flip Clock
            OdometerTimerView           // Odometer
*/

///////////////////////////////////////////////////////////////////////////////////////////////////
//
// TimerWidget :: Widget
//
///////////////////////////////////////////////////////////////////////////////////////////////////

var TimerWidget = Class.create(Widget, {
    widgetIdentifier: "com-apple-iweb-widget-timer",

    // Note -- order matters in the following arrays.  Must agree with
    // the enum values in BLTimerWidgetInfo.h.
    types: ['text', 'digital', 'odometer', 'flip'],

    // This is the box height correction factor to accommodate fancy fonts like "Big Caslon".
    // It should always be syncronized with 'line-height' CSS property in TimerWidget.css.
    minFontFactor: 1.2,

    initialize: function($super, instanceID, widgetPath, sharedPath, sitePath, preferences, runningInApp) {
        if(instanceID != null)
        {
            $super(instanceID, widgetPath, sharedPath, sitePath, preferences, runningInApp);
            this.autosizing = 0;
            this.view = null;

            this.units = [
                this.localizedString("Secs"),
                this.localizedString("Mins"),
                this.localizedString("Hours"),
                this.localizedString("Days"),
                this.localizedString("Years")];

            // Initialize default prefs.
            // According to the Countdown Widget HIS, v.1.7 the default
            // settings for the countdown widget are:
            //  Style: Digital
            //  Display:            Days, Hours, Minutes, Seconds
            //  Labels:             Displayed (checkbox selected)
            //  Date:               Today's date
            //  Time:               Current time 
            //  Height:             50px
            //  Digits Displayed:   00:00:00:00
            //  Labels:             Default CSS body paragraph style

            var targetDate = new Date();
            targetDate.setDate(targetDate.getDate());
            this.initializingDefaultPreferences = true;
            this.initializeDefaultPreferences({
                'type':             1,                              // 0 -Text; 1 - Digital; 2 - Odometer; 3 - Flip
                'target':           targetDate.getTime(),
                'unitsRange':       {'location': 1, 'length': 4},   // ddd:hh:mm:ss
                'labelsEnabled':    1
            });
            this.initializingDefaultPreferences = undefined;

            this.updateFromPreferences();
        }
    },
    
    onload: function() {
        // This method is called after Widget.initialize().
        // Therefore, we already updated from properties, and it's safe to create a view.
        this.view || this.p_createView();
    },

    onunload: function() {
        // Clear the refresh interval, if set one.
        this.refreshInterval === undefined || this.refreshInterval == 0 || clearInterval(this.refreshInterval);
        this.view = null;
    },

    changedPreferenceForKey: function(key) {
        if (!this.initializingDefaultPreferences && (key == 'type' || key == 'target' || key == 'unitsRange' || key == 'labelsEnabled'))
        {
            // For any of these properties changed we neen to re-create a view.
            this.updateFromPreferences();
            this.p_createView();
        }

        if (key == 'labelFontSize')
        {
            if (this.view)
            {
                this.view.updateFromLabelFontSize();
            }
        }
        else if (key == 'labelFontFactor')
        {
            if (this.view)
            {
                this.view.updateFromLabelFontFactor();
            }
        }
    },

    setDocumentInternalCSS: function($super, css) {
        $super(css);

        var currentFontFamily = this.p_currentFontFamily();
        if (this.lastFontFamily != currentFontFamily) {
            this.lastFontFamily = currentFontFamily;

            var testTable = new Element('table');
            var testRow = new Element('tr', {className: 'Timer_Labels'});
            var testCell = new Element('td');
            testTable.appendChild(testRow);
            testRow.appendChild(testCell);
            testTable.setStyle({position: 'absolute', top: '0px'});
            testRow.setStyle({fontSize: '100px'});
            testCell.innerText = '0123456789:';
            this.div().appendChild(testTable);
            var fontFactor = Math.max(testTable.offsetHeight / 100, this.minFontFactor);
            this.div().removeChild(testTable);
            this.setPreferenceForKey(fontFactor, 'labelFontFactor');
        }
    },

    updateFromPreferences: function() {
        this.targetDate = this.p_targetDateFromPreferences();
        this.unitsRange = this.p_unitsRangeFromPreferences();
        this.labelsEnabledValue = this.p_labelsEnabledFromPreferences();
    },

    // Timer widget specific.

    p_currentFontFamily: function() {
        var family = undefined;
        var labelElement = this.div().selectFirst('.Timer_Labels');
        if (labelElement) {
            family = labelElement.getStyle('font-family');
        }
        return family;
    },

    p_targetDateFromPreferences: function() {
        var target = this.preferenceForKey('target');
        return new Date(target);
    },

    p_unitsRangeFromPreferences: function() {
        var rangeDict = this.preferenceForKey('unitsRange');
        return new IWRange(rangeDict.location, rangeDict.length);
    },
    
    p_labelsEnabledFromPreferences: function() {
        return ((this.preferenceForKey('labelsEnabled') > 0) ? true : false);                        
    },

    currentTimeDelta: function() {
        // return the number of milliseconds by which the dates differ
        var now = new Date();
        return Math.max(0, this.targetDate - now);
    },
    
    p_createView: function() {
        // Try to create a view of the same height as the previous one (if exists).
        // Otherwise, create a view for a timer of default height=50px.
        // NOTE. This argument specifies the height of a timer, not a view.
        // A view may extend downwards to accommodate labels.
        var timerDisplayHeight = this.view != null ? this.view.timerDisplayHeight : -1;
 
        viewType = this.preferenceForKey('type');

        // Clear the refresh interval if set.
        this.refreshInterval === undefined || this.refreshInterval == 0 || clearInterval(this.refreshInterval);
        this.view = null;

        switch(this.types[viewType])
        {
            case 'odometer':
                this.view = new OdometerTimerView(this, timerDisplayHeight);
                break;
            case 'flip':
                this.view = new FlipTimerView(this, timerDisplayHeight);
                break;
            case 'digital':
                this.view = new DigitalTimerView(this, timerDisplayHeight);
                break;
            case 'text':
                this.view = new TextualTimerView(this, timerDisplayHeight);
                break;
            default:
                iWLog('timer widget could not instatiate unknown view');
                this.view = null;
        }

        this.view.setSelectedStyle();                                                                

        // Set window 'onresize' handler when constructor finished its work completely.
        if(this.runningInApp)
        {
            // When the user resizes a widget by dragging the sizing knobs, don't preserve the height.
            // Instead, size the timer to fit into the frame.
            // This works well because the timer widget's drawable info geometry has its aspect ratio locked.
            window.onresize = function()
            {
                if(this.autosizing == 0)
                {
                    this.view.resize(/*in_preserveHeight=*/ false);
                }
            }.bind(this);

            // Now, since new view is complemeted, it is safe to inform app of changed size.
            this.view.resizeWidgetWindow();
        }

        // Refresh the view once now, and set an interval to refresh regularly.
        this.view.refresh();
        this.refreshInterval = setInterval(this.view.refresh.bind(this.view), 1000);
    }
});


///////////////////////////////////////////////////////////////////////////////////////////////////
//
// TimerView
//
///////////////////////////////////////////////////////////////////////////////////////////////////

var TimerView = Class.create({

    // The CSS class of the labels, which is inspectable
    labelClassName: 'Timer_Labels',
    
    initialize: function(widget, in_timerDisplayHeight) {
        if(widget)
        {
            this.widget = widget;
            this.div = widget.div();

            this.p_calculateRatios();

            // Store the inital widget aspect ratio as provided by the app.
            var widgetWidth = toPixels(this.div.getStyle('width'));
            var widgetHeight = toPixels(this.div.getStyle('height'));
            this.widgetAspectRatio = widgetHeight / widgetWidth;

            this.handlingHandleResize = 0;

            // Remove children, if left from the previous view instance.
            var timer = this.widget.getElementById('timer');
            if (timer !== undefined) {
                timer.update();
            }

            this.timerDisplayHeight = in_timerDisplayHeight;

            // Number of digits required to represent each of date/time units.
            // Call this.p_setNumberOfDigitsPerUnit() to update these values from
            // the current granularity, specified by 'unitsRange' widget property.
            this.yDigitsNumber = 0;
            this.dDigitsNumber = 0;
            this.hDigitsNumber = 0;
            this.mDigitsNumber = 0;
            this.sDigitsNumber = 0;
            this.p_setNumberOfDigitsPerUnit();
        }
    },
    
    // The widget begins calling this method repeatedly using 'setInterval' function
    // as soon as it creates the appropriate timer view. The view specific rendering
    // should be implemented in the 'render' method of derived view classes.
    refresh: function() {
        this.render();
    },
    
    render: function() {
        // virtual
    },

    ensureVisible: function() {
        // Ensure that the timer is visible now.
        var timer = this.widget.div().selectFirst('.timer_widget');
        if (timer)
        {
            timer.setStyle({ visibility: 'visible' });
        }
    },
    
    setSelectedStyle: function() {
        // Tell the inspectors that we're editing the timer labels.  This allows the
        // user to set paragraph and character styling on the label text of the timer.
        this.widget.setPreferenceForKey(this.labelClassName, "x-selected-style-class", false);
    },
    
    updateFromLabelFontSize: function() {
        var labelFontSize = this.widget.preferenceForKey("labelFontSize");

        // Calculate timer display height from a givel labels font size,
        // resize the widget and inform the app about new size.
        this.timerDisplayHeight = this.fontFactor * this.digitsToLabelsFontsRatio * labelFontSize;
        this.resize(/*in_preserveHeight=*/ true);

        if(this.handlingHandleResize == 0)
        {
            this.resizeWidgetWindow();
        }
    },

    updateFromLabelFontFactor: function() {
        this.p_calculateRatios();
        this.updateFromLabelFontSize();
    },

    setLabelFontSize: function(labelFontSize)
    {
        labelFontSize >= 10 || (labelFontSize = 10);
        labelFontSize <= 50 || (labelFontSize = 50);
        var oldLabelFontSize = this.widget.preferenceForKey("labelFontSize");
        if (this.widget.runningInApp && (this.handlingHandleResize || !oldLabelFontSize) && !this.widget.preferences.isUndoingOrRedoing())
        {
            if (labelFontSize != oldLabelFontSize)
            {
                this.widget.setPreferenceForKey(labelFontSize, "labelFontSize", !!oldLabelFontSize);
            }
        }
    },

    //     There are 2 different types of resizing when running in app:
    //     1) resize when a widget property changed, and
    //     2) resize as result of mouse dragging of a window frame.
    // 
    //     1) If resized as result of a property changes, we MUST preserve the original timer display height.
    //     Then we calculate the entire widget width and height based on the timer dispaly height and properties.
    //     Finally we notify app about new widget size. In response the app must change a widget's drawable size,
    //     and if width has been changed, reposition it on a page horizontally to pin its center.
    //     In this case argument 'in_preserveHeight' must be 'true'.
    // 
    //     NOTE: The problem is that whenever a property changes we usually create a new view. Then 'resize'
    //     method is being called from within new view constructor, which means that 'window.onresize' is still bound
    //     to the previous view. Therefore, we call notification from [TimerWidget p_createView] after 'window.onresize'
    //     event handler is properly bound.
    // 
    //     2) If resized while a window frame being dragged, we completely rely on a frame size and position,
    //     provided by the app. In this case argument 'in_preserveHeight' must be 'false'.

    resize: function(in_preserveHeight) {
        // If a widget resizes because of a property change, then we calculate widget
        // dimensions based on a given timer display height in 'resizeTimer' method.
        // Otherwise (in case of a frame dragging), we should use widget dimensions,
        // provided by the app. The problem is that at this point a widget's 'width' and
        // 'height' properties are not updated yet, and we need to pre-calculate them here.
        if (in_preserveHeight) {
            // Store new widget dimensions.
            // NOTE: Window size is provided by the app. Its dimensions are always integers,
            // while widget dimenstions ('width' and 'height' properties of a widget's <div>)
            // may be float, provided that window.dimension = Math.ceil(widget.dimension).
            if (this.widget.runningInApp) {
                this.windowWidth = window.innerWidth;
                this.windowHeight = window.innerHeight;
            }
            else {
                this.windowWidth = Math.ceil(this.widget.div().offsetWidth);
                this.windowHeight = Math.ceil(this.widget.div().offsetHeight);
            }

            // A widget preference change case: do resize the timer first, provided that the height is preserved.
            var widgetSize = this.resizeTimer(/*in_preserveHeight=*/ true);
        
            // Store widget dimensions, calculated from the timer display height.
            this.widgetWidth = widgetSize.width;
            this.widgetHeight = widgetSize.height;
            this.widgetAspectRatio = this.widgetHeight / this.widgetWidth;
        }
        else {
            ++this.handlingHandleResize;

            try {
                var widthChanged = false, heightChanged = false;

                // Store new widget dimensions.
                // NOTE: Window size is provided by the app. Its dimensions are always integers,
                // while widget dimenstions ('width' and 'height' properties of a widget's <div>)
                // may be float, provided that window.dimension = Math.ceil(widget.dimension).
                if(this.windowWidth != window.innerWidth)
                {
                    this.windowWidth = window.innerWidth;
                    widthChanged = true;
                }
                if(this.windowHeight != window.innerHeight)
                {
                    this.windowHeight = window.innerHeight;
                    heightChanged = true;
                }

                if(widthChanged || heightChanged)
                {
                    // Dragging a frame case: pre-calculate widget dimensions first.
                    var windowAspectRatio = this.windowHeight / this.windowWidth;
                    if (windowAspectRatio > this.widgetAspectRatio) {
                        this.widgetWidth = this.windowWidth;
                        this.widgetHeight = this.windowWidth * this.widgetAspectRatio;
                    }
                    else if (windowAspectRatio < this.widgetAspectRatio) {
                        this.widgetWidth = this.windowHeight / this.widgetAspectRatio;
                        this.widgetHeight = this.windowHeight;
                    }
                    else {
                        this.widgetWidth = this.windowWidth;
                        this.widgetHeight = this.windowHeight;
                    }
                // Do resize the timer to fit into pre-calculated widget dimensions.
                    this.resizeTimer(/*in_preserveHeight=*/ false);
                }
            }
            finally {
                --this.handlingHandleResize;
            }
        }
    },

    resizeWidgetWindow: function() {
        ++this.widget.autosizing;
        try
        {
            this.widget.setPreferenceForKey({
                "width": this.widgetWidth, 
                "height": this.widgetHeight,
                "centerHor": true
            }, "x-widgetSize", false);   
        }
        finally
        {
            --this.widget.autosizing;
        }
    },

    resizeTimer: function(in_preserveHeight) {
    },

    timeUnits: function(ms, unitsRange) {
        (function(){
            return 0 <= unitsRange.min() && unitsRange.min() <= unitsRange.max() && unitsRange.max() <= 5;
        }).assert("Invalid time units range (min="+unitsRange.min()+", max="+unitsRange.max()+")");

        var result = [0, 0, 0, 0, 0];
        var divisors = [365, 24, 60, 60, 1000];
        var unitsRangeMin = unitsRange.min();

        var acc = ms;
        for(i=4; i>=unitsRangeMin; i--) {
            result[i] = Math.floor(acc/divisors[i]);
            acc /= divisors[i];
            if(i>unitsRangeMin) {
                result[i] = result[i] % divisors[i-1];
            }
        }

        return {y:result[0], d:result[1], h:result[2], m:result[3], s:result[4]};
    },

    // Useful helper methods.

    // Assuming that given in_height is the height of a pure digits box,
    // return the total height of a widget (digits box + labels box, if present).
    heightWithLabels: function(in_height) {
        return this.widget.labelsEnabledValue ? in_height * (1 + this.labelsToDigitsFontsRatio) : in_height;
    },

    // Assuming that given in_height is the total height of a widget (digits box + labels box, if present), 
    // return the height of a pure digits box.
    heightWithoutLabels: function(in_height) {
        return this.widget.labelsEnabledValue ? in_height / (1 + this.labelsToDigitsFontsRatio) : in_height;
    },

    // Returns an array of digits (integers in range 0...9),
    // which represents a given integer value ('in_value').
    // If 'in_length' > 0, then returns exactly 'in_length' digits,
    // padding zeros at the left, if necessary.
    // Otherwise, returns as many digits as necessary to represent
    // a given value, but at least one.
    // Example: getDigitsFromInteger(12345, 8) returns digits = [0,0,0,1,2,3,4,5],
    // where digits[0] = 0 and digits[7] = 5.
    getDigitsFromInteger: function(in_value, in_length) {
        (function(){return in_value >= 0 && in_length >= 0;}).assert("getDigitsFromInteger - Invalid arguments: in_value="+in_value+" in_length="+in_length);
        var digits = [];
        var digitCount = 0;
        var ratio = in_value;
        do {
            var mod = this.intMod(ratio, 10);
            digits.push(mod.remainder);
            ratio = mod.ratio;
            digitCount++;
        } while((digitCount < 2) || (in_length == 0 && ratio != 0) || (in_length > 0 && digitCount < in_length));
        return digits.reverse();
    },

    // For given value ('in_value') and divider ('in_divider')
    // returns their integer ratio and remainder. 
    intMod: function(in_value, in_div) {
        (function(){return in_value >= 0 && in_div > 0;}).assert("intMod - Invalid arguments: in_value="+in_value+" in_div="+in_div);
        var ratio = Math.floor(in_value / in_div);
        return {ratio: ratio, remainder: Math.floor(in_value % in_div)};
    },

    // Private methods specific for this class.

    p_calculateRatios: function() {
        // The useful fonts and box metric ratios. These formulas result from the HIS requirement:
        //    The minimum size of the font in the label is 10 points. 
        //    The font will be 10pt when the widget is scaled to its smallest size - 50px in height.
        // NOTE: We assume that the specified widget size (50px) includes labels.
        this.fontFactor = this.widget.preferenceForKey('labelFontFactor') || this.widget.minFontFactor;
        this.digitsToLabelsFontsRatio = (5.0 / this.fontFactor) - 1;
        this.labelsToDigitsFontsRatio = 1 / this.digitsToLabelsFontsRatio;

        // Inform the app about the relative labels height.
        if (this.widget.runningInApp) {
            this.widget.setPreferenceForKey(this.labelsToDigitsFontsRatio, 'x-labelsToDigitsFontsRatio', false);
        }
    },

    // Call this method on view initialization and whenever the timer granularity changes
    // to calculate the number of digits required to represent each of date/time units.
    p_setNumberOfDigitsPerUnit: function() {
        // Calculate the current time delta.
        var ms = this.widget.currentTimeDelta();

        // Split time delta into units according to the currently set date/time granularity.
        var unitsRange = this.widget.unitsRange;
        var unitsRangeMin = unitsRange.min();   // The leftmost unit.
        var timeUnits = this.timeUnits(ms, unitsRange);

        // Store number of digits required for each selected unit.
        this.yDigitsNumber = unitsRange.containsLocation(0) ? this.getDigitsFromInteger(timeUnits.y, 0).size() : 0;
        this.dDigitsNumber = unitsRange.containsLocation(1) ? this.getDigitsFromInteger(timeUnits.d, 1 == unitsRangeMin ? 0 : 3).size() : 0;
        this.hDigitsNumber = unitsRange.containsLocation(2) ? this.getDigitsFromInteger(timeUnits.h, 2 == unitsRangeMin ? 0 : 2).size() : 0;
        this.mDigitsNumber = unitsRange.containsLocation(3) ? this.getDigitsFromInteger(timeUnits.m, 3 == unitsRangeMin ? 0 : 2).size() : 0;
        this.sDigitsNumber = unitsRange.containsLocation(4) ? this.getDigitsFromInteger(timeUnits.s, 4 == unitsRangeMin ? 0 : 2).size() : 0;
    },

    labelsRowMarkup: function() {
        var labelsRowMarkupString = "";
        if (this.widget.labelsEnabledValue > 0) {
            var unitsRange = this.widget.unitsRange;
            var unitsRangeMin = unitsRange.min();   // The leftmost unit.
            var unitsRangeMax = unitsRange.max();   // The rightmost unit.
            for(i=(4-unitsRangeMin); i>(4-unitsRangeMax); i--) {
                labelsRowMarkupString += "<td style='text-align: center;' class='fg-label'>" + this.widget.units[i] + "</td>";
                if (i > (4-unitsRangeMax) + 1) {
                    labelsRowMarkupString += "<td class='fg-label-separator'></td>"; // row corresponding to separator (if it exists)
                }
            }
        }
        return labelsRowMarkupString;
    }
});


///////////////////////////////////////////////////////////////////////////////////////////////////
//
// TextualTimerView :: TimerView
//
///////////////////////////////////////////////////////////////////////////////////////////////////

var TextualTimerView = Class.create(TimerView, {
    timerType: 'text',

    initialize: function($super, widget, in_timerDisplayHeight) {
        $super(widget, in_timerDisplayHeight);

        // Create a string of table elements for digits
        var unitsRange = this.widget.unitsRange;
        var timeDigitColumns = "";
        for (i = unitsRange.min(); i < unitsRange.max(); i++) {
            // Add digit
            timeDigitColumns += "<td></td>";
            // Add separator
            if (unitsRange.containsLocation(i + 1)) {
                timeDigitColumns += '<td>:</td>';
            }
        }
        
        var template = new Template("<table border='0' id='#{table_id}'><tr id='#{digit_id}' class='#{label_class}'>#{columns}</tr>" +
            "<tr id='#{labels_id}' class='#{labels_class}'>#{labels_markup}</tr></table>");

        this.widget.getElementById('timer').update(template.evaluate({
            table_id: this.widget.getInstanceId("timer_block"),
            digit_id: this.widget.getInstanceId("timer_digit_block"),
            label_class: this.labelClassName,
            columns: timeDigitColumns,
            labels_id: this.widget.getInstanceId("timer_labels_block"),
            labels_class: this.labelClassName,
            labels_markup: this.labelsRowMarkup()
        }));

        this.widget.getElementById("timer_digit_block").selectFirst('td').setStyle({textAlign: 'right'});
    
        this.resize(/*in_preserveHeight=*/ true);
    },
    
    // Implements Textual Timer specific rendering.
    render: function($super) {
        var needsUpdateSize = false;
        var ms = this.widget.currentTimeDelta();        
        var unitsRange = this.widget.unitsRange;

        var timerDigitBlock = this.widget.getElementById("timer_digit_block");

        // Digits row in the table
        var timeUnits = this.timeUnits(ms, unitsRange);
        var unitsArray = [timeUnits.y, timeUnits.d, timeUnits.h, timeUnits.m, timeUnits.s];
        var unitsPaddingArray = [1, 3, 2, 2, 2];
        var timerDigitColumns = timerDigitBlock.select("td");
        for (i = unitsRange.min(); i < unitsRange.max(); i++)
        {
            var minLength = 2;
            if(i > unitsRange.min())
            {
                minLength = Math.max(minLength, unitsPaddingArray[i]);
            }
            timerDigitColumns[(i - unitsRange.min()) * 2].update(unitsArray[i].toPaddedString(minLength, 10));
        }

        this.ensureVisible();
    },

    resizeTimer: function( in_preserveHeight ) {
        // For a given height the width of a text timer varies depending on font properties.
        // Therefore, we can not pre-calculate the width here based on its height solely,
        // rather we have to render the text timer and then get its width from a browser.

        // Get the timer's display height and total height.
        var timerTotalHeight = 0;
        var timerDisplayHeight = 0;
        if (in_preserveHeight) {
            // A widget preference change case: preserve the height, the width may vary.
            // NOTE. At this point widget dimensions may not be known yet (e.g. a widget is just unarchived).
            if (this.timerDisplayHeight > 0) {
                // Scale to preserve the current timer's height.
                timerTotalHeight = this.heightWithLabels(this.timerDisplayHeight);
                timerDisplayHeight = this.timerDisplayHeight;
            }
            else if (this.widgetHeight > 0) {
                // Scale to fit the current widget's height.
                timerTotalHeight = this.widgetHeight; 
                timerDisplayHeight = this.heightWithoutLabels(this.widgetHeight);
            } 
            else {
                // Scale to fit the current window's height.
                timerTotalHeight = this.windowHeight; 
                timerDisplayHeight = this.heightWithoutLabels(this.windowHeight);
            }
        } 
        else {
            // Dragging a frame case: we do have pre-calculated widget dimensions at this point.
            // Scale to fit the pre-calculated widget's height.
            timerTotalHeight = this.widgetHeight; 
            timerDisplayHeight = this.heightWithoutLabels(this.widgetHeight);
        }

        // Update labels font size and notify the app, if necessary.
        // NOTE: timerDisplayHeight does NOT include labels, only digits!
        var baseLabelFontSize = timerDisplayHeight / (this.digitsToLabelsFontsRatio * this.fontFactor);
        this.setLabelFontSize(baseLabelFontSize);

        // read a new value from our preferences in case our suggestion wasn't taken
        baseLabelFontSize = this.widget.preferenceForKey("labelFontSize");

        // Labels on or off - resize to match
        var timerDigitBlock = this.widget.getElementById("timer_digit_block");
        timerDigitBlock.setStyle({fontSize: px(this.digitsToLabelsFontsRatio * baseLabelFontSize)});

        var timerLabelsBlock = this.widget.getElementById("timer_labels_block");
        if (this.widget.labelsEnabledValue > 0) {
            timerLabelsBlock.setStyle({fontSize: px(baseLabelFontSize)});
        } else {
            timerLabelsBlock.setStyle({fontSize: "0"});
        }

        this.timerDisplayHeight = timerDisplayHeight;

        this.render();

        // check if the widget changed size due to this rendering
        var sizeChanged = false;
        var timerTable = this.widget.getElementById("timer_block");
        if (timerTable && timerTable.offsetWidth > 0 && timerTable.offsetHeight > 0)
        {
            if ((this.totalNativeWidth === undefined) ||    // new TimerView
                (this.totalNativeHeight === undefined) ||   // new TimerView
                (this.totalNativeWidth != timerTable.offsetWidth) ||
                (this.totalNativeHeight != timerTable.offsetHeight))
            {
                this.totalNativeWidth = timerTable.offsetWidth;
                this.totalNativeHeight = timerTable.offsetHeight;

                // We now know the new proportions of the timer table; ask the app to resize the widget.
                this.widgetWidth = this.totalNativeWidth;
                this.widgetHeight = this.totalNativeHeight;
                this.widgetAspectRatio = this.widgetHeight / this.widgetWidth;

                sizeChanged = true;
            }
        }

        if (sizeChanged && this.handlingHandleResize == 0 && this.widget.view === this) {
            this.resizeWidgetWindow();
        }

        return new IWSize(this.totalNativeWidth, this.totalNativeHeight);
    }
});


///////////////////////////////////////////////////////////////////////////////////////////////////
//
// GraphicalTimerView :: TimerView
//
///////////////////////////////////////////////////////////////////////////////////////////////////

/*
    The base view class for all graphical timers. It provides the framework for building markup
    and rendering a graphical timer. This framework uses virtual methods, which should be overridden
    in derived classes. Also several useful helper methods are provided.
*/

var GraphicalTimerView = Class.create(TimerView, {

    initialize: function($super, widget, in_timerDisplayHeight) {
        $super(widget, in_timerDisplayHeight);

        // Preload assests.
        this.preloadAssets();

        // At this point we already know the number of digits
        // required to represent each of the date/time units.

        // Make initial markup.
        var markup = "<div class='graphical_timer'>";
        markup += "<div class='graphical_timer_background'>";
        markup += this.makeBGMarkup();
        markup += "</div>";
        markup += "<div class='graphical_timer_foreground'>";
        markup += this.makeFGMarkup();
        markup += "</div>";
        markup += "<div class='graphical_timer_labels'>";
        markup += this.makeLabelMarkup();
        markup += "</div>";
        markup += "</div>";
        var timerElement = this.widget.getElementById('timer'); // NOTE: The actual id is this.widget.instanceID+'-timer', e.g. 'widget0-timer'.
        timerElement.update(markup);

        // Set up background images. They won't change for a given type of a timer.
        this.setBGImageSources();

        // Calculate native (unscaled) geometry of the timer. This won't change for a given type of a timer either.
        this.setNativeGeometry();

        // Resize a timer to fit actual geometry and render it.
        this.resize(/*in_preserveHeight=*/ true);

        // NOTE: Foreground images will be regulary updated in the 'render' method called repeatedly by the timer.
    },
    
    // Virtual methods. MAY be overridden in derived classes to add functionality (call base class method!).

    preloadAssets: function() {
        // Preload digits. Instance var 'this.timerType' specifies individual folder for each type of a timer view.
        this.digits = [
            {image: IWCreateImage(this.assetPath('0', this.widget.widgetPath+'/'+this.timerType+'/0.pdf')), size: new IWSize(120, 150)},
            {image: IWCreateImage(this.assetPath('1', this.widget.widgetPath+'/'+this.timerType+'/1.pdf')), size: new IWSize(120, 150)},
            {image: IWCreateImage(this.assetPath('2', this.widget.widgetPath+'/'+this.timerType+'/2.pdf')), size: new IWSize(120, 150)},
            {image: IWCreateImage(this.assetPath('3', this.widget.widgetPath+'/'+this.timerType+'/3.pdf')), size: new IWSize(120, 150)},
            {image: IWCreateImage(this.assetPath('4', this.widget.widgetPath+'/'+this.timerType+'/4.pdf')), size: new IWSize(120, 150)},
            {image: IWCreateImage(this.assetPath('5', this.widget.widgetPath+'/'+this.timerType+'/5.pdf')), size: new IWSize(120, 150)},
            {image: IWCreateImage(this.assetPath('6', this.widget.widgetPath+'/'+this.timerType+'/6.pdf')), size: new IWSize(120, 150)},
            {image: IWCreateImage(this.assetPath('7', this.widget.widgetPath+'/'+this.timerType+'/7.pdf')), size: new IWSize(120, 150)},
            {image: IWCreateImage(this.assetPath('8', this.widget.widgetPath+'/'+this.timerType+'/8.pdf')), size: new IWSize(120, 150)},
            {image: IWCreateImage(this.assetPath('9', this.widget.widgetPath+'/'+this.timerType+'/9.pdf')), size: new IWSize(120, 150)}];
        this.digits.each(function(digit) { digit.image.load(); });

        // Pluck out the widest size from the digit array.
        this.digitSize = $(this.digits).pluck('size').inject(new IWSize(0, 0), function(acc, size) {
            return size.width > acc.width ? size : acc;
        });

        // Preload separator
        this.separator = {image: IWCreateImage(this.assetPath('fg-separator', this.widget.widgetPath+'/'+this.timerType+'/fg-separator.pdf')),
                          size: new IWSize(50, 150)};
        this.separator.image.load();

        // Preload left cap.
        this.leftCap = {image: IWCreateImage(this.assetPath('bg-cap-left', this.widget.widgetPath+'/'+this.timerType+'/bg-cap-left.pdf')),
                        size: new IWSize(5, 45)};
        this.leftCap.image.load();
        
        // Preload right cap.
        this.rightCap = {image: IWCreateImage(this.assetPath('bg-cap-right', this.widget.widgetPath+'/'+this.timerType+'/bg-cap-right.pdf')),
                         size: new IWSize(5, 45)};
        this.rightCap.image.load();

        // Preload center fill.
        this.center = {image: IWCreateImage(this.assetPath('bg-center', this.widget.widgetPath+'/'+this.timerType+'/bg-center.pdf')),
                       size: new IWSize(30, 45)};
        this.center.image.load();
    },

    makeBGMarkup: function() {
        var markup = "";
        markup += "<img id='"+this.widget.instanceID+"-bg-cap-left' src='' />";
        markup += "<img id='"+this.widget.instanceID+"-bg-center' src='' />";
        markup += "<img id='"+this.widget.instanceID+"-bg-cap-right' src='' />";
        return markup;
    },

    makeFGMarkup: function() {
        var markup = "";
        var digit = "<img class='fg-digit' src='' />";
        var separator = imgMarkup(this.separator.image.sourceURL(), "", "class='fg-separator' ", "", false);

        // Group all foreground elements to make center positioning easier.
        markup += "<div class='graphical_digits_placeholder'>";
        markup += digit.times(this.yDigitsNumber);
        markup += separator.times(this.yDigitsNumber > 0 && this.dDigitsNumber > 0);
        markup += digit.times(this.dDigitsNumber);
        markup += separator.times(this.dDigitsNumber > 0 && this.hDigitsNumber > 0);
        markup += digit.times(this.hDigitsNumber);
        markup += separator.times(this.hDigitsNumber > 0 && this.mDigitsNumber > 0);
        markup += digit.times(this.mDigitsNumber);
        markup += separator.times(this.mDigitsNumber > 0 && this.sDigitsNumber > 0);
        markup += digit.times(this.sDigitsNumber);
        markup += "</div>";

        return markup;
    },

    setBGImageSources: function() {
        setImgSrc(this.widget.getElementById('bg-cap-left'), this.leftCap.image.sourceURL(), false);
        setImgSrc(this.widget.getElementById('bg-cap-right'), this.rightCap.image.sourceURL(), false);
        setImgSrc(this.widget.getElementById('bg-center'), this.center.image.sourceURL(), false);
    },

    setNativeGeometry: function() {
        var maxHeight = this.leftCap.size.height;   // Assume that this.leftCap.size.height == this.rightCap.size.height
        var digitHeightRatio = 0.55;                // The height of a digit relative to the height of a timer.
        var digitScaleFactor = (maxHeight / this.digitSize.height) * digitHeightRatio;

        // Get display units involved.
        this.numberOfDigits = this.widget.div().select('img.fg-digit').size();
        this.numberOfSeparators = this.widget.div().select('img.fg-separator').size();

        // Calculate and store native (unscaled) geometry.
        this.digitNativeHeight = this.digitSize.height * digitScaleFactor;
        this.digitNativeWidth = this.digitSize.width * digitScaleFactor;
        this.separatorNativeWidth = this.separator.size.width * (this.textSeparator ? digitScaleFactor : 1);
        this.separatorNativeHeight = this.separator.size.height * (this.textSeparator ? digitScaleFactor : 1);
        this.leftCapNativeWidth = this.leftCap.size.width;
        this.rightCapNativeWidth = this.rightCap.size.width;
        this.totalNativeHeight = this.heightWithLabels(maxHeight);
        this.totalNativeWidth = 
            this.leftCapNativeWidth + 
            this.digitNativeWidth * this.numberOfDigits + 
            this.separatorNativeWidth * this.numberOfSeparators + 
            this.rightCapNativeWidth;
    },

    // Useful helper methods.

    assetPath: function($super, asset, defaultPath) {
        return this.widget.preferenceForKey(asset) || defaultPath;
    },
    
    fitTimerIntoView: function(in_nativeTimerWidth, in_nativeTimerHeight, in_viewWidth, in_viewHeight, in_preserveHeight) {
        // NOTE: Both 'in_nativeTimerHeight' and 'in_viewHeight' arguments include labels height, if labels present.
        var timerWidth = in_viewWidth;
        var timerHeight = in_viewHeight;
        var sf = 1;

        // The problem with preserving the height is that when we call this method first time
        // for the newly created view, the height is undefined yet - it is to be calculated.
        // That's why we need to pass this height over from the old view to the new one.
        // However, if this is the very first view for this widget, we just scale
        // to fit the current window height.
        if (in_preserveHeight) {
            if (this.timerDisplayHeight > 0) {
                // Scale to preserve the current timer's height.
                var timerDisplayHeight = this.timerDisplayHeight;
                sf = this.heightWithLabels(timerDisplayHeight)/in_nativeTimerHeight;
                timerHeight = in_nativeTimerHeight * sf;
                timerWidth = in_nativeTimerWidth * sf;
            } else {
                // Scale to fit the current window's height.
                timerHeight = this.windowHeight;
                sf = timerHeight/in_nativeTimerHeight;
                timerWidth = in_nativeTimerWidth * sf;
            }
        } else {
            // Scale to fit the current view (either width or height).
            var viewAspectRatio = in_viewHeight / in_viewWidth;
            var timerAspectRatio = in_nativeTimerHeight / in_nativeTimerWidth;
            if (viewAspectRatio < timerAspectRatio) {
                // Scale to fit the view's height.
                sf = timerHeight/in_nativeTimerHeight;
                timerWidth = in_nativeTimerWidth * sf;
            }
            else {
                // Scale to fit the view's width.
                sf = timerWidth/in_nativeTimerWidth;
                timerHeight = in_nativeTimerHeight * sf;
            }
        }
        
        return {width:timerWidth, height:timerHeight, scale:sf};
    },

    centerTimerInView: function(in_timerWidth, in_timerHeight, in_labelHeight, in_labelOffsetLeft) {        
        var widgetDiv = this.widget.div();
        var graphical_timer = widgetDiv.select('div.graphical_timer')[0];
        var timerLeft = 0;
        graphical_timer.setStyle({
            width:  px(in_timerWidth),
            height: px(in_timerHeight),
            left:   px(timerLeft)
        });

        var labelTable = this.widget.getElementById("timer_labels_table");
        if (labelTable) {
            // Implicitly, since the table exist, labels must be enabled.
            labelTable.setStyle({
                left: px(timerLeft + in_labelOffsetLeft),
                top:  px(in_timerHeight)
            });
        }
    },

    setFGDigits: function(io_placeholders, in_pos, in_len, in_value) {
        var digits = this.getDigitsFromInteger(in_value, in_len);
        for (var i = 0; i < in_len; ++i) {
            var img = io_placeholders[in_pos + i];
            var digit = digits[i];
            if(img.digit != digit)
            {
                img.digit = digit; // cache to avoid setting the url multiple times
                setImgSrc(img, this.digits[digit].image.sourceURL(), false);
            }
            if(!img.visible())
            {
                img.show();
            }
        }
    },

    makeLabelMarkup: function() {
        var labelMarkup = "";
        if (this.widget.labelsEnabledValue > 0) {
            labelMarkup = "<table border='0' class='" + this.labelClassName + "' id='" +
                this.widget.getInstanceId("timer_labels_table") +
                "' style='position: absolute; left:0; top: 0; width:100%; height:100%;'>" +
                "<tr>" +
                this.labelsRowMarkup() +
                "</tr>" +
                "</table>";
        }
        return labelMarkup;
    },

    resizeTimer: function(in_preserveHeight) {
        // Resize the background here
        var thisWidth = this.widgetWidth;
        var thisHeight = this.widgetHeight;

        var timerScaledSize = this.fitTimerIntoView(this.totalNativeWidth, this.totalNativeHeight, thisWidth, thisHeight, in_preserveHeight);
        var sf = timerScaledSize.scale;
        var timerWidth = timerScaledSize.width;
        var timerHeight = this.heightWithoutLabels(timerScaledSize.height);

        var leftPos = 0;
        var bgElementWidth = 0;

        var leftCap = this.widget.getElementById('bg-cap-left');
        var rightCap = this.widget.getElementById('bg-cap-right');
        var center = this.widget.getElementById('bg-center');

        var capWidth = Math.floor(this.leftCap.size.width*sf);

        // Resize background: the left cap.
        var leftCapElement = this.widget.getElementById('bg-cap-left');
        leftCapElement.setStyle({
            height: px(timerHeight), 
            width:  px(capWidth), 
            left:   px(leftPos)
        });
        leftPos += capWidth;

        // Resize background: central area (the background for all digits and separators).                                    
        var centerElement = this.widget.getElementById('bg-center');
        var width = Math.floor(timerWidth - 2*capWidth);
        centerElement.setStyle({
            height: px(timerHeight), 
            width:  px(width), 
            left:   px(leftPos)
        });
        leftPos += width;

        // Resize background: the right cap.
        var rightCapElement = this.widget.getElementById('bg-cap-right');
        rightCapElement.setStyle({
            height: px(timerHeight), 
            width:  px(capWidth), 
            left:   px(leftPos)
        });
        leftPos += capWidth;
        
        // Resize foreground: placeholders for digits and separators.
        leftPos = capWidth; // start just to the right of the left cap
        var digitHeight = this.digitNativeHeight * sf;
        var digitWidth = this.digitNativeWidth * sf;
        var separatorHeight = this.textSeparator ? this.separatorNativeHeight * sf : timerHeight;
        var separatorWidth = this.separatorNativeWidth/this.separatorNativeHeight * separatorHeight;

        var centerWidth =  (this.digitNativeWidth * this.numberOfDigits + this.separatorNativeWidth * this.numberOfSeparators) * sf;
        var placeholder = this.widget.div().select('div.graphical_digits_placeholder')[0];
        placeholder.setStyle({
            left:   px((timerWidth - centerWidth) / 2),
            height: px(timerHeight)
        });

        leftPos = 0;
        placeholder.select('img').each(function(element){
            var height = element.hasClassName('fg-digit') ? digitHeight : separatorHeight;
            var width = Math.max(element.hasClassName('fg-digit') ? digitWidth : separatorWidth, 1);
            var top = (timerHeight - height) / 2;
            element.setStyle({
                height: px(height), 
                top: px(top), 
                left: px(leftPos), 
                width: px(width)
            });
            leftPos += width;
        });

        // Resize labels table.
        var labelHeight = this.resizeLabels(timerScaledSize.height, digitWidth, separatorWidth);

        // Center the timer in the view.
        this.centerTimerInView(timerWidth, timerHeight, labelHeight, placeholder.offsetLeft);

        // Store the current timer display height.
        this.timerDisplayHeight = timerHeight;

        this.render();

        return new IWSize(timerScaledSize.width, timerScaledSize.height); 
    },

    resizeLabels: function(areaHeight, digitWidth, separatorWidth) {
        var leftMostDigitLeft = 0;
        var labelHeight = 0;
        var labelTable = this.widget.getElementById("timer_labels_table");

        // Figure out what the font size should be provided that, according
        // to HIS, min timer display height=50px and font height=10pt.
        // Implicitly, since the table exist, labels must be enabled.
        // NOTE: 'areaHeight' is the total widget height which includes labels, if present.
        var labelFontSize = areaHeight / (((labelTable ? 1 : 0) + this.digitsToLabelsFontsRatio) * this.fontFactor);
        this.setLabelFontSize(labelFontSize);
        labelHeight = labelFontSize * this.fontFactor;

        if (labelTable) {
            labelFontSize = this.widget.preferenceForKey("labelFontSize");
            labelTable.setStyle({
                fontSize:   px(labelFontSize)
            });
            // Size the width of each table cell
            var labelsFG = labelTable.select('td.fg-label');
            var curLeft = leftMostDigitLeft;
            var curField = this.widget.unitsRange.min();
            var nDigitsArray = [this.yDigitsNumber, this.dDigitsNumber, this.hDigitsNumber, this.mDigitsNumber, this.sDigitsNumber];
            var digitIndex = 0;
            for (var i = this.widget.unitsRange.min(); i < this.widget.unitsRange.max(); i++) {
                var width = 0;
                var digitsInThisField = nDigitsArray[i];
                var fieldLeft = curLeft;
                for (var j = 0; j < digitsInThisField; j++) {
                    curLeft += digitWidth;
                    digitIndex++;
                }
                labelsFG[i - this.widget.unitsRange.min()].setStyle({ 
                    width: px(curLeft - fieldLeft) });
                if (i < this.widget.unitsRange.max() - 1) {
                    curLeft += separatorWidth;
                }
            }
            labelTable.setStyle({ width: px(curLeft - leftMostDigitLeft) });

            // Size all separator cells also
            labelTable.select('td.fg-label-separator').each(function(separatorCell) {
                separatorCell.setStyle({ width: px(separatorWidth) });
            });
        }

        return labelHeight;
    },

    render: function($super) {
        // Get the time delta.
        var ms = this.widget.currentTimeDelta();

        // Get the time delta in units according to specified date/time granularity.
        var timeUnits = this.timeUnits(ms, this.widget.unitsRange);

        // Get the placeholders.
        var digitImages = this.widget.div().select('img.fg-digit');
        var separatorImages = this.widget.div().select('img.fg-separator');
        
        var digitCount = 0;
        var separatorCount = 0;
        var unitsCount = 0;

        // Set years digits.
        if (this.yDigitsNumber > 0) {
            this.setFGDigits(digitImages, digitCount, this.yDigitsNumber, timeUnits.y);
            digitCount += this.yDigitsNumber;
        }
        // Set separator.
        if (this.yDigitsNumber > 0 && this.dDigitsNumber > 0) {
            this.p_setFGSeparator(separatorImages, separatorCount);
            ++separatorCount;
        }
        // Set days digits.
        if (this.dDigitsNumber > 0) {
            this.setFGDigits(digitImages, digitCount, this.dDigitsNumber, timeUnits.d);
            digitCount += this.dDigitsNumber;
        }
        // Set separator.
        if (this.dDigitsNumber > 0 && this.hDigitsNumber > 0) {
            this.p_setFGSeparator(separatorImages, separatorCount);
            ++separatorCount;
        }
        // Set hours digits.
        if (this.hDigitsNumber > 0) {
            this.setFGDigits(digitImages, digitCount, this.hDigitsNumber, timeUnits.h);
            digitCount += this.hDigitsNumber;
        }
        // Set separator.
        if (this.hDigitsNumber > 0 && this.mDigitsNumber > 0) {
            this.p_setFGSeparator(separatorImages, separatorCount);
            ++separatorCount;
        }
        // Set minutes digits.
        if (this.mDigitsNumber > 0) {
            this.setFGDigits(digitImages, digitCount, this.mDigitsNumber, timeUnits.m);
            digitCount += this.mDigitsNumber;
        }
        // Set separator.
        if (this.mDigitsNumber > 0 && this.sDigitsNumber > 0) {
            this.p_setFGSeparator(separatorImages, separatorCount);
            ++separatorCount;
        }
        // Set seconds digits.
        if (this.sDigitsNumber > 0) {
            this.setFGDigits(digitImages, digitCount, this.sDigitsNumber, timeUnits.s);
            digitCount += this.sDigitsNumber;
        }

        // Call it now to ensure that the timer is visible.
        this.ensureVisible();
    },
    
    // Private methods specific for this class.

    p_setFGSeparator: function(io_placeholders, in_pos) {
        var img = io_placeholders[in_pos];
        if(!img.visible())
        {
            img.show();
        }
    }
    
});


///////////////////////////////////////////////////////////////////////////////////////////////////
//
// OdometerTimerView :: GraphicalTimerView
//
///////////////////////////////////////////////////////////////////////////////////////////////////

var OdometerTimerView = Class.create(GraphicalTimerView, {
    timerType: 'odometer',

    initialize : function($super, widget, in_timerDisplayHeight) {
        this.textSeparator = false;
        return $super(widget, in_timerDisplayHeight);
    },

    preloadAssets: function($super) {
        $super();
        this.leftCap.size = new IWSize(9, 50);
        this.rightCap.size = new IWSize(9, 50);
        this.center.size = new IWSize(38, 50);
        this.separator.size = new IWSize(5, 50);
        
        $A(this.digits).each(function(digit) {
            digit.size = new IWSize(210, 280);
        });
    }
    
});


///////////////////////////////////////////////////////////////////////////////////////////////////
//
// FlipTimerView :: GraphicalTimerView
//
///////////////////////////////////////////////////////////////////////////////////////////////////

var FlipTimerView = Class.create(GraphicalTimerView, {
    timerType: 'flip',

    initialize : function($super, widget, in_timerDisplayHeight) {
        this.textSeparator = false;
        return $super(widget, in_timerDisplayHeight);
    },

    preloadAssets: function($super) {
        $super();
        this.leftCap.size = new IWSize(8, 50);
        this.rightCap.size = new IWSize(8, 50);
        this.center.size = new IWSize(39, 50);
        this.separator.size = new IWSize(1, 50);

        $A(this.digits).each(function(digit) {
            digit.size = new IWSize(39, 50);
        });
    }
    
});


///////////////////////////////////////////////////////////////////////////////////////////////////
//
// DigitalTimerView :: GraphicalTimerView
//
///////////////////////////////////////////////////////////////////////////////////////////////////

var DigitalTimerView = Class.create(GraphicalTimerView, {
    timerType: 'digital',

    initialize : function($super, widget, in_timerDisplayHeight) {
        this.textSeparator = true;
        $super(widget, in_timerDisplayHeight);
    },

    preloadAssets: function($super) {
        $super();
        this.leftCap.size = new IWSize(5, 45);
        this.rightCap.size = new IWSize(5, 45);
        this.center.size = new IWSize(30, 45);
        this.separator.size = new IWSize(50, 150);

        $A(this.digits).each(function(digit) {
            digit.size = new IWSize(120, 150);
        });
    }
    
});

