//
//  iWeb - YouTube.js
//  Copyright (c) 2008 Apple Inc. All rights reserved.
//


var YouTubeWidget = Class.create(Widget, {

    widgetIdentifier: "com-apple-iweb-widget-YouTube",
    thumbnailURL: null,

    initialize: function($super, instanceID, widgetPath, sharedPath, sitePath, preferences, runningInApp)
    {
        if (instanceID)
        {
            $super(instanceID, widgetPath, sharedPath, sitePath, preferences, runningInApp);
            
            if(this.runningInApp)
            {
                window.onresize = this.resize.bind(this);
            }

            var parentDiv = this.div("youTube");
            this.m_views = {};
            this.m_views["movie"] = new YouTubeMovieView(this, parentDiv);
            if (runningInApp)
            {
                this.m_views["no-movie-status"] = new YouTubeNoMovieStatus(this, parentDiv);
                this.m_views["invalid-url-status"] = new YouTubeInvalidURLStatus(this, parentDiv);
                this.m_views["user-offline-status"] = new YouTubeUserOfflineStatus(this, parentDiv);
            }
            else
            {
                this.m_views["no-movie-status"] = new YouTubePublishedErrorStatus(this, parentDiv);
                this.m_views["invalid-url-status"] = this.m_views["no-movie-status"];
                this.m_views["user-offline-status"] = this.m_views["no-movie-status"];
            }

            this.showView("no-movie-status");
            
            this.setPreferenceForKey(false, "x-snapshotAvailable", false);

            // other parameter processing
            this.updateFromPreferences();
        }
    },

    changedPreferenceForKey: function(key)
    {
        var widgetDiv = this.div();

        if (this.preferenceForKey("x-online") === false)
        {
            this.showView("user-offline-status");
        }
        else if ((key == 'showRelatedVideos') || (key == 'address'))
        {
            this.updateFromPreferences();
        }
        else if (key == 'x-thumbnailMode')
        {
            if (this.thumbnailURL != null)
            {
                if (this.preferenceForKey(key))
                {
                    this.m_views["movie"].showThumbnail(true);
                }
                else
                {
                    this.m_views["movie"].showThumbnail(false);
                }
            }
            else
            {
                // We don't switch to thumbnail mode because thumbnail URL is not set up.
                // This will happen if we're offline, or if the video doesn't have an Atom
                // entry for some reason (I've seen this happen). In this case the normal
                // snapshot mechanism will take a snapshot of the YouTube widget, which
                // includes the controller.
            }
        }
    },

    updateFromPreferences: function()
    {
        if (!this.normalizingAddress)
        {
            this.normalizingAddress = true;
            var specifiedAddress = this.preferenceForKey("address");
            var viewToShow = "no-movie-status";
    
            if (specifiedAddress && (specifiedAddress.length > 0))
            {
                var movieURL = this.preferenceForKey("movieURL");
    
                // Start by processing the address preference.  Is the specified address different than the
                // last normalized one we reflected back to the HUD?  If not, avoid an infinite loop by 
                // skipping this part.  If so, the user must have changed the address field, so update the
                // widget's address state.  Also, scan other parameters like color and border.
                if (specifiedAddress != this.preferenceForKey('lastNormalizedAddress')) 
                {
                    // Clear movieURL; attempt to rebuild it here.
                    movieURL = null;
                    this.setPreferenceForKey(0, "x-mediaDuration", false);
                    
                    var youTubeURL = this.p_scrapeYouTubeURLFromString(specifiedAddress);
                    var videoGUID = this.p_scrapeVideoGUIDFromString(youTubeURL);
                    var params = youTubeURL.toQueryParams();
                
                    if (videoGUID && videoGUID.length > 0)
                    {
                        // Copy properties we care about to a new subset of query parameters
                        var newQueryParams = {};
                        ['color1', 'color2', 'border'].each(function(property) {
                            var value = params[property];
                            if (value) 
                            {
                                newQueryParams[property] = value;
                            }
                        });

                        // Construct a full URL for the movie including the GUID and all parameters
                        // we were able to glean from the incoming URL
                        movieURL = "http://www.youtube.com/v/" + videoGUID;
                        var queryParamString = $H(newQueryParams).toQueryString();
                        movieURL += ((queryParamString.length == 0) ? "" : ("&" + queryParamString));
                         if (movieURL != this.preferenceForKey("movieURL"))
                         {
                                this.redoThumbnail(videoGUID);
                         }
                         
                        // Create a 'normalized address' that contains the movie GUID but actually links
                        // to the page on YouTube that shows this movie.  Show that address in the HUD,
                        // as that's what users are used to seeing.
                        var normalizedAddress = "http://www.youtube.com/watch?v=" + videoGUID;
                        this.setPreferenceForKey(normalizedAddress, "lastNormalizedAddress", false);
                        this.setPreferenceForKey(normalizedAddress, "address", false);

                        // If URL contains 'rel' parameter, then override the 'show related videos' flag in the HUD.
                        var rel = params['rel'];
                        if (rel !== undefined)
                        {
                            this.setPreferenceForKey((rel !== "0"), "showRelatedVideos", false);
                        }
 
                        // Assume a snapshot is available.
                        this.setPreferenceForKey(true, "x-snapshotAvailable", false);
                    }
                }
                
    
                // Now whether or not we updated the address, update the movie URL to reflect
                // the 'show related videos' flag in the HUD. To suppress related videos the URL
                // should contain parameter 'rel=0', otherwise related videos will show up.
                var showRelatedVideos = this.preferenceForKey("showRelatedVideos");
                if (movieURL && (showRelatedVideos !== undefined))
                {
                    var tempQueryParams = $H(movieURL.httpURLQueryString().toQueryParams());
                    if (showRelatedVideos)
                    {
                        tempQueryParams.unset("rel");
                    }
                    else
                    {
                        tempQueryParams.set("rel", "0");
                    }
                    movieURL = movieURL.split("&")[0];
                    var queryParamString = tempQueryParams.toQueryString();
                    movieURL += ((queryParamString.length == 0) ? "" : ("&" + queryParamString));
                }
    
                if (movieURL != this.preferenceForKey("movieURL"))
                {
                    this.setPreferenceForKey(movieURL, "movieURL", false);
                }
                     
                var viewToShow = (movieURL && movieURL.isHTTPURL && movieURL.isHTTPURL() ? "movie" : "invalid-url-status");
                this.m_views[viewToShow].render();
            }
            this.m_currentViewName = viewToShow;
            this.showView(viewToShow);
            this.normalizingAddress = undefined;
        }
    },

    redoThumbnail: function(videoGUID)
    {
         this.thumbnailURL = null;
         
         var thumbnailURL = null;
         // Go get the thumbnail image for the movie.
         // todo(brentm): We don't need to do this for Widgets that don't go in a blog media placeholder,
         // so we should avoid getting this if we can.
         new Ajax.Request("http://gdata.youtube.com/feeds/api/videos/" + videoGUID, {
            method: 'get',
            onFailure: function(request)
            {
                this.setPreferenceForKey(false, "x-snapshotAvailable", false);
                this.render();
            }.bind(this),
            onSuccess: function(request)
            {
                var entryNode = ajaxGetDocumentElement(request);
                var mediaNS = "http://search.yahoo.com/mrss/";
                var youTubeNS = "http://gdata.youtube.com/schemas/2007";
                var mediaGroup = entryNode.getElementsByTagNameNS(mediaNS, "group");
                if (mediaGroup && mediaGroup[0])
                {
                    var maxHeight = 0;
                    var mediaThumbnails = mediaGroup[0].getElementsByTagNameNS(mediaNS, "thumbnail");
                    $A(mediaThumbnails).each(function(element)
                    {
                        var height = parseInt(element.getAttribute("height"));
                        if (height > maxHeight)
                        {
                            maxHeight = height;
                            thumbnailURL = element.getAttribute("url");
                        }
                    }.bind(this));
                    
                    this.m_views["movie"].setThumbnailURL(thumbnailURL);
                    this.thumbnailImage = IWCreateImage(thumbnailURL);
                    this.thumbnailImage.load(function()
                    {
                        this.thumbnailURL = thumbnailURL;
                        this.setPreferenceForKey(true, "x-snapshotAvailable", false);
                        this.setPreferenceForKey(true, "x-snapshotReady", false);
                    }.bind(this));

                    // Attempt to glean the duration to display in the My Albums page
                    var duration = mediaGroup[0].getElementsByTagNameNS(youTubeNS, "duration");
                    if (duration && duration[0])
                    {
                        var seconds = parseFloat(duration[0].getAttribute("seconds"));
                        if (seconds > 0)
                        {
                            this.setPreferenceForKey(seconds, "x-mediaDuration", false);
                        }
                    }
                }
            }.bind(this)
        });
    },

    resize: function()
    {
        $H(this.m_views).each(function(pair) {
            pair.value.resize();
        } );
    },

    onload: function()
    {
        if (!this.runningInApp)
        {
        }        
    },

    onunload: function()
    {
    },
    
// Private implementation

    p_scrapeYouTubeURLFromString: function(specifiedAddress)
    {
        var address = specifiedAddress;

        // Check for embed code; extract URL if found.
        var match = specifiedAddress.match(/(value|src)\s*\=\s*([']([^']+)[']|["]([^"]+)["])/);
        if(match && match.length == 5)
        {
            address = match[3] || match[4];
        }
    
        // We should now have a valid URL.  Strip whitespace and return it.
        return address.strip();
    },
    
    p_scrapeVideoGUIDFromString: function(specifiedAddress)
    {
        var videoGUID  = null;

        // Check for a video GUID on its own
        // Alphanumeric plus hyphen and underscore, and contains at least one digit.
        var match = specifiedAddress.match(/^([a-zA-Z0-9_\-]*\d[a-zA-Z0-9_\-]*)$/);
        if(match)
        {
            videoGUID = match[1];
        }
        else
        {
            match = specifiedAddress.match(/^(http[s]?:\/\/)?(\w+\.)?youtube\.com(\/.+)$/);
            if(match && match.length == 4)
            {
                var pathAndParams = match[3];
                var params = specifiedAddress.toQueryParams();
            
                var videoGUID = params["v"];
                if(videoGUID == null)
                {
                    // Check for /v/<videoGUID> format
                    match = pathAndParams.match(/^\/v\/([^\/&\?]+)/);
                    if(match)
                    {
                        videoGUID = match[1];
                    }
                }
            }
        }
        
        return videoGUID;
    }

});

var YouTubeMovieView = Class.create(View, {
    
    m_divId: "movie",
    m_divClass: "YouTubeMovie",
    m_thumbnailURL: "",
    
    setThumbnailURL: function(url)
    {
        if (this.m_thumbnailURL != url)
        {
            this.m_thumbnailURL = url;
            this.render();
        }
    },
    
    render: function()
    {
        // defer drawing until widget is loaded
        if(this.preferences && this.preferences.postNotification)
        {
            this.m_widget.preferences.postNotification("BLWidgetIsSafeToDrawNotification", 0);
        }
        
        var markup = '<img style="visibility: hidden; position: absolute; width:100%; height:100%;" src="'+ this.m_thumbnailURL + '" />' +
                        '<object style="visibility: visible; position:absolute; width:100%; height:100%">' +
                           '<param name="movie" value="' + this.m_widget.preferenceForKey('movieURL') + '"></param>' +
                           '<param name="wmode" value="transparent">' +
                            '<embed src="' + this.m_widget.preferenceForKey('movieURL') + '" type="application/x-shockwave-flash" wmode="transparent" width="100%" height="100%"></embed>'+
                    '</object>';
        
        this.ensureDiv().update(markup);

        if(this.runningInApp)
        {
            setTimeout(function() 
            {
                if(this.preferences && this.preferences.postNotification)
                {
                    this.m_widget.preferences.postNotification("BLWidgetIsSafeToDrawNotification", 1);
                }
                if (this.m_widget.preferenceForKey("x-snapshotAvailable") === false)
                {
                    this.m_widget.setPreferenceForKey(true, "x-timeElapsedAfterMovieLoad", false);
                }
            }.bind(this), 1000);
        }
    },
    
    showThumbnail: function(flag)
    {
        var theImg = this.ensureDiv().select('img')[0];
        var theObj = this.ensureDiv().select('object')[0];
        if (theImg && theObj)
        {
            if (flag)
            {
                theObj.setStyle({visibility: "hidden"});
                theImg.setStyle({visibility: "visible"});
            }
            else
            {
                theImg.setStyle({visibility: "hidden"});
                theObj.setStyle({visibility: "visible"});
            }
        }
    }
});

var YouTubeNoMovieStatus = Class.create(StatusView, {
    m_divId: "no-movie-status",
    m_divClass: "YouTubeStatusView",

    badgeImage: "youtube-placeholder.png",
    badgeImageWidth: 128,
    badgeImageHeight: 69
});

var YouTubeInvalidURLStatus = Class.create(StatusView, {
    m_divId: "invalid-url-status",
    m_divClass: "YouTubeStatusView",

    badgeImage: "youtube-placeholder_disabled.png",
    badgeImageWidth: 128,
    badgeImageHeight: 69,

    statusMessageKey: "<b>The YouTube URL you entered is invalid.</b><br />Double-check the URL on YouTube, and then try again.",
    upperRightBadge: "error-glyph.png",
    upperRightBadgeWidth: 24,
    upperRightBadgeHeight: 19
});

var YouTubeUserOfflineStatus = Class.create(StatusView, {
    m_divId: "user-offline-status",
    m_divClass: "YouTubeStatusView",

    badgeImage: "youtube-placeholder_disabled.png",
    badgeImageWidth: 128,
    badgeImageHeight: 69,

    statusMessageKey: "<b>You must be connected to the Internet to view the YouTube movie.</b>",

    upperRightBadge: "error-glyph.png",
    upperRightBadgeWidth: 24,
    upperRightBadgeHeight: 19
});

var YouTubePublishedErrorStatus = Class.create(StatusView, {
    m_divId: "published-error-status",
    m_divClass: "YouTubeStatusView",

    badgeImage: "youtube-placeholder.png",
    badgeImageWidth: 128,
    badgeImageHeight: 69
});
