;+
; 
; SMOOTHMAP,  mapin, psfin, VARIN=varin, $
;             PSF_X0=psf_x0, PSF_Y0=psf_y0, MAXLIKELY=maxlikely, $
;             SMMAP=smmap, SMVAR=smvar, DOUBLE=double
;
; Cross-correlate map with psf. Optionally filter variance
; map. Optionally use maximum likely formulation of cross-correlation,
; weighting by variance map.
;
;
; PARAMETERS:
;
;   mapin:     input signal map [2-d array]
;   psfin:     input psf to with which to smooth map [2-d array]
;
;
; OPTIONAL INPUTS:
;
;   VARIN:     input variance map [2-d array]
;   PSF_X0:    X pixel position of centre of psf in psf map (default is
;                middle rounded down)
;   PSF_Y0:    Y pixel position of centre of psf in psf map (default is
;                middle rounded down)
;   MAXLIKELY: if set, use maximum likely formulation
;   CORRELATE: if set, flip PSF for mathematical def of
;                cross-correlation, else performs convolution
;   DOUBLE:    set DOUBLE keyword in calls to FFT
;
;
; OUTPUTS:
;
;   SMMAP:     smoothed signal map
;   SMVAR:     smoothed variance map
;
;
; 2008-05-22 gmarsden@phas.ubc.ca
;
; HISTORY:
;
;   2007-07-22 Add CORRELATE keyword
;
;-

PRO SMOOTHMAP, mapin, psfin, VARIN=varin, $
               PSF_X0=psf_x0, PSF_Y0=psf_y0, MAXLIKELY=maxlikely, $
               SMMAP=smmap, SMVAR=smvar, CORRELATE=correlate, DOUBLE=double

IF KEYWORD_SET(maxlikely) THEN domaxlikely = 1 ELSE domaxlikely = 0
IF KEYWORD_SET(varin)   THEN donoise     = 1 ELSE donoise     = 0

s = SIZE(mapin, /DIM)
mnx = s[0]
mny = s[1]

s = SIZE(psfin, /DIM)
pnx = s[0]
pny = s[1]
IF NOT KEYWORD_SET(psf_x0) THEN psf_x0 = pnx/2 ELSE psf_x0 = ROUND(psf_x0)
IF NOT KEYWORD_SET(psf_y0) THEN psf_y0 = pny/2 ELSE psf_y0 = ROUND(psf_y0)

IF KEYWORD_SET(correlate) THEN BEGIN
    ; rotate psf and shift centre
    psf = ROTATE(psfin, 2)
    px0 = pnx - 1 - psf_x0
    py0 = pny - 1 - psf_y0
ENDIF ELSE BEGIN
    ; don't rotate for convolution
    psf = psfin
    px0 = psf_x0
    py0 = psf_y0
ENDELSE

; pad psf
psfpad = FLTARR(mnx, mny)
psfpad[0:pnx-1,0:pny-1] = psf

; shift psf so that centre is at (0,0)
psfpad = SHIFT(psfpad, -px0, -py0)



; perform convolutions
IF domaxlikely THEN BEGIN
    denom   = REAL_PART( FFT(FFT(1/varin, DOUBLE=double) * $
                             FFT(psfpad^2, DOUBLE=double), $
                             /INV, DOUBLE=double) ) * mnx * mny
    smmap   = REAL_PART( FFT(FFT(mapin/varin, DOUBLE=double) * $
                             FFT(psfpad, DOUBLE=double), $
                             /INV, DOUBLE=double) ) / denom * mnx * mny
    smvar = 1.0 / denom
ENDIF ELSE BEGIN
    smmap   = REAL_PART( FFT(FFT(mapin, DOUBLE=double) * $
                             FFT(psfpad, DOUBLE=double), $
                             /INV, DOUBLE=double) ) * mnx * mny

    IF donoise THEN $
      smnoise = REAL_PART( FFT(FFT(varin, DOUBLE=double) * $
                               FFT(psfpad^2, DOUBLE=double), $
                               /INV, DOUBLE=double) * mnx * mny )
ENDELSE


END
